function [ark,Basis,C] = NumericalRank(A,tol,HL)

% <Purpose>
%   Computing the numerical rank of a matrix.
%
% <Syntax>
%   [r,Basis,C] = NumericalRank(A,tol,HL);
%
% <Input Parameters>
%   1.   A -- the target matrix.
%   2. tol -- (optional) the rank decision threshold.
%                        default: tol = sqrt(n)*norm(A,1)*eps.
%   3.  HL -- (optional) string parameter
%       If HL = 'high rank', call the high rank-revealing algorithm.
%       If HL = 'low rank', call the low rank-revealing algorithm.
%       default: HL = 'high rank'.
%
% <Output Parameters>
%   1. r     -- the numerical rank of A.
%   2. Basis -- (optional)
%       When running the high rank-revealing algorithm, its columns
%       form an orthonormal basis of the numerical kernel;
%       When running the low rank-revealing algorithm, its columns
%       form an orthonormal basis of the numerical range;
%   3.     C -- (optional)
%       Matlab cell array contains data required by updating/downdating.
%       When running the high rank-revealing algorithm,
%       C{1,1} = rank : the numerical rank of A;
%       C{2,1} = Basis : an orthonormal basis of the numerical kernel;
%       C{3,1} = Q : the Q in the kernel stacked QR factorization of A;
%       C{4,1} = R : the R in the kernel stacked QR factorization of A;
%       C{5,1} = tau : the scaling factor in the kernel stacked matrix;
%       C{6,1} = tol : the rank decision threshold;
%       When running the low rank-revealing algorithm,
%       C{1,1} = rank : the numerical rank of A;
%       C{2,1} = U : U in the USV+E decomposition of A;
%       C{3,1} = V : V in the USV+E decomposition of A;
%       C{4,1} = S : S in the USV+E decomposition of A;
%       C{5,1} = tol : the rank decision threshold;
%
% <Examples>
%   >>  [r,Basis,C] = NumericalRank(A,1e-12)
%   >>  [r,Basis,C] = NumericalRank(A,1e-8,'low rank')
%

if (nargin < 2)
    n = size(A,2);
    tol = sqrt(n)*norm(A,1)*eps;
end
if (nargin < 3)
    HL = 'high rank';
end
if ( ~strcmp(HL,'high rank') && ~strcmp(HL,'low rank') )
    error('The third input should be "high rank" or "low rank". ')
end

if ( strcmp(HL,'high rank') )
    if (nargout == 1)
        [ark] = harank(A,tol);
    elseif (nargout == 2)
        [ark,Basis] = harank(A,tol);
    elseif (nargout == 3)
        [ark,Basis,Q,R,tau] = harank(A,tol);
        C{1,1} = ark;
        C{2,1} = Basis;
        C{3,1} = Q;
        C{4,1} = R;
        C{5,1} = tau;
        C{6,1} = tol;
        C{1,2} = 'rank';
        C{2,2} = 'kernel';
        C{3,2} = 'Q';
        C{4,2} = 'R';
        C{5,2} = 'tau';
        C{6,2} = 'tol';
    end
elseif ( strcmp(HL,'low rank') )
    if (nargout == 1)
        [ark] = larank(A,tol);
    elseif (nargout ==2)
        [ark,Basis] = larank(A,tol);
    elseif (nargout == 3)
        [ark,Basis,S,V] = larank(A,tol);
        C{1,1} = ark;
        C{2,1} = Basis;
        C{3,1} = V;
        C{4,1} = S;
        C{5,1} = tol;
        C{1,2} = 'rank';
        C{2,2} = 'U : range';
        C{3,2} = 'V : row space';
        C{4,2} = 'S';
        C{5,2} = 'tol';
    end
end;
%-----------------------------------------------------------------------
% End of function harank
%-----------------------------------------------------------------------

function [ark,NB,Q,R,tau] = harank(A,tol)

% <Description>
%   the high rank-revealing algorithm
%
% <Syntax>
%   [ark,NB,Q,R,tau] = harank(A,tol);
%
% <Input Parameters>
%   1.   A -- the target matrix. (m>=n)
%   2. tol -- the rank decision threshold.
%             ( default: tol = sqrt(n)*norm(A,1)*eps )
%
% <Output parameters>
%   1. ark -- the numerical rank of A.
%   2.  NB -- an orthonormal basis for the numerical kernel.
%   3.   Q -- the Q in the kernel stacked QR factorization of A.
%   4.   R -- the R in the kernel stacked QR factorization of A.
%   5. tau -- the scaling factor. ( sqrt(n)*norm(A,1) )

if (nargin < 1)
    error('Not enough input arguments.')
end
[m,n] = size(A);
if (m < n)
    error('The system is underdetermined; Use the transpose of A.')
end

tau = sqrt(n)*norm(A,'inf');       %  the scalar of the deflation.
if (nargin == 1)
    tol = tau*eps;
end

if (tau < 1e-7)                    % Case: A is too small.
    ark = 0;
    w = randn(n,1);
    tmp = 1/norm(w);
    w = tmp*w;
    NB = eye(n) - (2*w)*w';
    B = [tau*NB.';A];
    [Q,R] = qr(B);
    return;
end
%------------------------------
ark = n;
tmpNB = zeros(n,n);
if (nargout > 2)
    [Q,R] = qr(A);
else
    R = qr(A); R = triu(R(1:n,:));
end;

nrow = m;
for kk = 1:n
    [w,s] = singular_n(R,tau,tol);
    if (s > tol)
        break;
    end
    ark = ark - 1;
    tmpNB(:,n-ark) = w;
    if (nargout > 2)
        [R,trans] = hesqr([tau*w';R]);
        Q = [ [1,zeros(1,nrow)]; [zeros(nrow,1),Q]];
        for j = 1:n
            T = [trans(:,j),[-conj(trans(2,j));conj(trans(1,j))]];
            Q(:,j:j+1) = Q(:,j:j+1)*T';
        end;
    else
        R = hesqr([tau*w';R]);
    end
    nrow = nrow + 1;
end;

nul = n - ark;
if (nul == 0)
    NB = [];
else
    NB = zeros(n,nul);
    for k = 1:nul
        NB(:,k) = tmpNB(:,nul-k+1);
    end
end
%-----------------------------------------------------------------------
% End of function harank
%-----------------------------------------------------------------------

function [v,s] = singular_n(R,scale,tol)

% <Purpose>
%   Computing the smallest singular value of an upper triangular matrix.
%
% <Syntax>
%   [v,s] = sigular_n(R,scale,tol);
%
% <Input parameters>
%   1.     R  -- an upper triangular matrix.
%   2. scale  -- the norm of R.
%   3.   tol  -- the rank decision threshold.
%
% <Output Parameters>
%   1.     v  -- the smallest singular vector.
%   2.     s  -- the smallest singular value.

n = size(R,2);
t1 = tol/sqrt(n);
t2 = max(1.0,scale)*eps*sqrt(n);
ztol = max(t1,t2);

v = randn(n,1);
tmp = 1/norm(v);
v = tmp*v;
u = zeros(1,n);
max_iter = 8;
for k = 1:n
    if (abs(R(k,k)) < eps)
        R(k,k) = eps;
    end
end;

for k = 1:max_iter
    %--------------------------- forward elimination
    u(1) = v(1)/R(1,1);
    for kk = 2:n
        u(kk) = (v(kk) - u(1:kk-1)*R(1:kk-1,kk))/R(kk,kk);
    end;
    tmp = 1/norm(u);
    u = tmp*u;
    %--------------------------- backward substitution
    v(n) = u(n)/R(n,n);
    for kk = (n-1):-1:1
        v(kk) = (u(kk) - R(kk,kk+1:n)*v(kk+1:n))/R(kk,kk);
    end;
    s = 1/norm(v);
    v = s*v;
    %---------------------------
    if (k > 1)
        if (abs(s - s_old)/s < ztol)
            break;
        end
        if (s < tol)
            break;
        end
    end;
    s_old = s;
end;
%-----------------------------------------------------------------------
% End of function
%-----------------------------------------------------------------------

function [A,trans] = hesqr(A)

% <Syntax>
%   [A,trans] = hesqr(A);
%
% <Input parameters>
%          A -- an upper Hessenberg matrix. (m >= n)
%
% <Output parameters>
%   1.     A -- an upper triangular matrix.
%   2. trans -- rotation used.

[~,n] = size(A);
trans = zeros(2,n);
for j = 1:n
    [G,d] = givensmt(A(j:j+1,j));
    if (abs(d) < eps)
        A(j:j+1,j) = [eps;0];
        trans(1:2,j) = [1;0];
    else
        if j < n
            A(j:j+1,j:n) = G*A(j:j+1,j:n);
        else
            A(n:n+1,n) = [d;0];
        end
        trans(:,j) = G(:,1);
    end
end
%-----------------------------------------------------------------------
% End of function
%-----------------------------------------------------------------------

function [G,d] = givensmt(v)

% <Purpose>
%   Computing the Givens matrix G such that G*v = [d;0].
%
% <Syntax>
%   [G,d] = givensmt(v);
%
% <Input Parameters>
%      v -- a target vector. ( 2-by-1 )
%
% <Output Parameters>
%   1. G -- the 2x2 Givens rotation matrix.
%   2. d -- G*v = [d;0].

d = norm(v,2);
if (abs(d) < eps)
    G = eye(2);
else
    c = v(1)/d; s = v(2)/d;
    G = [conj(c),conj(s);-s,c];
end;
%-----------------------------------------------------------------------
% End of function
%-----------------------------------------------------------------------

function [ark,U,S,V] = larank(A,tol)

% <Description>
%   the low rank-revealing algorithm
%
% <Syntax>
%   [ark,U,S,V] = larank(A,tol);
%
% <Input Parameters>
%   1.   A -- the target matrix.
%   2. tol -- the rank decision threshold.
%             ( default: tol = sqrt(n)*norm(A,1)*eps )
%
% <Output Parameters>
%   1.   ark -- the numerical rank of A.
%   2. U,S,V -- (optional) the USV+E decomposition of matrix A.

[m,n] = size(A);
scale = norm(A,1);
if (nargin == 1)
    tol = sqrt(n)*scale*eps;
end

U = zeros(m,m);
if (tol>1e-11)
    for ark = 0: n
        if (ark == 0)
            [u,s] = powerAAT(A,scale,tol);
        else
            [u,s] = implicitAAT(A,U,scale,tol);
        end
        if (s < tol), break; end;
        U(:,ark+1) = u;
    end
else
    B = A;
    for ark = 0: n
        [u,s,v] = powerAAT(B,scale,tol);
        if (s < tol), break; end;
        U(:,ark+1) = u;
        B = B - u*v';
    end
end
U(:,ark+1:m) = [];

if (nargout >= 3)
    V = (U'*A)'; [V,S] = qr(V,0); S = S';
end
%-----------------------------------------------------------------------
% End of function
%-----------------------------------------------------------------------

function [y,nrmy,x] = powerAAT(A,scale,tol)

% <Description>
%   Computing a unit vector in the numerical range of A.
%
% <Synopsis>
%   [u,s,v] = powerAAT(A,scale,tol);
%
% <Input parameters>
%   1.     A -- the target matrix.
%   2. scale -- the infinity norm of A.
%   3.   tol -- the rank decision tolerance.
%
% <Output Parameters>
%   1.     u -- a unit vector in the numerical range of A.
%   2.     s -- the 2-norm of A'*u.
%   3.     v -- v = u'*A.

[m,n] = size(A);
t1 = tol/sqrt(n);
t2 = max(1.0,scale)*eps*sqrt(n);
itol = max(t1,t2);
max_iter = 8;

y = randn(m,1);
nrmy = norm(y);
y = (1/nrmy)*y;
for k = 1:max_iter
    %-----------------------------------
    x = (y'*A)';
    nrmx = norm(x); x = (1/nrmx)*x;
    y = A*x;
    nrmy = norm(y);  y = (1/nrmy)*y;
    eta = (tol/nrmy)^(2*k);
    if (eta < t2), break; end
    
    if (k > 5)
        errnrm = abs(nrmy-nrmy0);
        if (errnrm < itol), break; end
        if (errnrm < t2*nrmy), break; end
    end
    nrmy0 = nrmy;
end

if (nargout == 3)
    x = (y'*A)';
end
%-----------------------------------------------------------------------
% End of function
%-----------------------------------------------------------------------

function [u,s] = implicitAAT(A,U,scale,tol)

% <Description>
%   Computing a unit vector in the numerical range of A-U*U'A.
%
% <Synopsis>
%   [u,s] = ImplicitAAT(A,U,scale,tol);
%
% <Input parameters>
%   1.     A -- the target matrix.
%   2.     U -- the partial range of A.
%   3. scale -- the infinity norm of A.
%   4.   tol -- the rank decision tolerance.
%
% <Output Parameters>
%   1.     u -- a unit vector in the numerical range of A.
%   2.     s -- the 2-norm of A'*u.

[m,n] = size(A);
t2 = max(1.0,scale)*eps*sqrt(n);
max_iter = 8;

z = randn(m,1);
cs = zeros(1,max_iter);

for k = 1:max_iter
    %-----------------------------------
    tu = (z'*U)';
    tv = U*tu;
    w = z - tv;
    p = (w'*A)';
    %-----------------------------------
    nrmp = norm(p); x = (1/nrmp)*p; cs(k) = nrmp;
    %-----------------------------------
    z = A*x;
    
    eta = (tol/nrmp)^(2*k-1);
    if (eta < t2), break; end
    
    if (k > 4)
        errnrm = abs(nrmp-nrmp0);
        if (errnrm < t2*nrmp), break; end
        %--------------------------------
        p0 = (2*cs(k-1)-cs(k)-cs(k-2));
        if (abs(p0) > eps)
            p1 = (cs(k)-cs(k-1))^2 /p0;
            if (p1 < t2), break; end
            if (cs(k)+p1 < tol), break; end
        end
    end
    nrmp0 = nrmp;
end

tu = (z'*U)';
tv = U*tu;
y = z - tv;
s = norm(y); u = (1/s)*y;
%-----------------------------------------------------------------------
% End of function
%-----------------------------------------------------------------------
