function [F,nF] = Fiter(X,param,swi,flag)  
% FITER Performs the computation of the next term for Examples 1 to 12
%       It gives the next iterate X_{i+1}=F(X_i) for the Examples
%       1 to 8, 11 and 12, or the new term of the partial sums for the
%       Examples 9 to 10 
%
% [F,nF] = Fiter(X,param,swi,flag) 
%
% Inputs:
%   X       Input term of the sequence
%   param   Cell array defining additional parameters needed.
%           For examples 1 to 5, param{1} contains the alpha
%           value. For the other ones, it contains other data needed.
%   swi     Example number (string)
%   flag    This value is needed only for Examples 8, 9, 10
%           If flag = 0 we initialize the persistent parameter Bn (Ex 8)
%                       we initialize the persistent parameters nterm,
%                       psum (Ex 9, 10) and also ind (Ex 9)
%                       we initialize the persistent parameters nterm (Ex
%                       11)
%           If flag ~=0 we compute only the value nF = |X - F(X)| (Ex 8)
%                       we compute only the value nF = |X - psum| (Ex 9,
%                       10)
%                       Not usable for example 11
%           Otherwise we implement the iterative algorithm of Z.-Z. Bai (Ex
%           8) or we continue to compute the partial sum (Ex 9 and 10)
%
% Outputs:
%   F       Next term of the sequence F(X_i) or new term of the partial sum
%   nF      Euclidean norm of F

% Authors: Claude Brezinski, Michela Redivo-Zaglia
% V1.0 - December 2016
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Values that needs to mantains (Examples 8, 9, 10, 11)
persistent ind nterm psum Bn

% Check the number of input parameters
if (nargin < 3) || (nargin > 4)
    error(' Fiter: Not enought or too many input parameters')
end

% Compute the first dimension 
n = size(X,1);

% Select the example asked
switch swi
    
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXEMPLE 1    Solution of a system of nonlinear equations (K.M. Brown)
%              Figure 1 of the paper
%
    case '1'
       
       F(1:n-1) = X(1:n-1)-(n+1);
       F = F + sum(X,1);
       F(n) = prod(X,1)-1;
       F=F(:);
       F = X +param{1}.*F;
       nF=norm(X-F);
       
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXEMPLE 2     Solution of a system of nonlinear equations. 
%               Figure 2 of the paper
%
    case '2'
        F = n - sum(cos(X)) + (1:n)'.*(1-cos(X)) - sin(X);
        F = X + param{1}.*F;
        F=F(:);
        nF=norm(X-F);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXEMPLE 3     Solution of a system of nonlinear equations. 
%               It is not possible to modify the dimension.
%               Figure 3 of the paper
%
    case '3'
        F(1)=(1-param{1}).*X(1)+param{1}.*(X(1)*X(2)^3/2-1/2+sin(X(3)));
        F(2)=(1-param{1}).*X(2)+param{1}.*(exp(1+X(1)*X(2))/2+1/2);
        F(3)=(1-param{1}).*X(3)+param{1}.*(1-cos(X(3))+X(1)^4-X(2));

        F=F(:);
        nF=norm(X-F);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXEMPLES 4    Solution of a system of nonlinear equations. This system 
%               has at least two solutions. 
%               It is not possible to modify the dimension.
%               (Figures 4 and 5 of the paper)
%
    case {'4a','4b'}
        F(1) = X(1)+param{1}.*(abs(X(1)^2-1)+X(2)-1);
        F(2) = X(2)+param{1}.*(X(2)^2+X(1)-2);
        F=F(:);
        nF=norm(X-F);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXEMPLE 5     Solution of a system of nonlinear equations (Grau Sanchez 
%               et al)
%               It is not possible to modify the dimension.
%               Figure 6 of the paper
%
    case '5'
        F = -X - exp(-X) + sum(X);
        F = X + param{1}.*F;
        F=F(:);
        nF=norm(X-F);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXEMPLE 6     Solution of a system of linear equations with several
%               right hand sides AX = B.
%               Figure 7 of the paper
%
    case '6'
        % Compute X = (I-A)X + B
        F = (eye(n,n)-param{1})*X + param{2};   
        nF = norm(X-F);
       
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXEMPLE 7     Solution of a matrix equation (C.-H. Guo)
%               Figure 8 of the paper
%
    case '7'  
        % Compute X = I+ A^*A - A^* X^-1 A
        F = param{2}-param{1}'*(X\param{1});
        nF = norm(X-F);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXEMPLE 8     Solution of a matrix equation (Z.-Z. Bai). 
%               The solution is known when the A_i = p^i A_0, A_0 given.
%               Otherwise the solution is unknown.
%               (Figure 9 of the paper)
%      
    case {'8a','8b'}  
        % Initialize Q to the identity matrix
        Q=eye(n,n);    
        % Define the number of terms of the infinity sum
        ninf = 100;
        % Compute Q = I - sum_i=1^inf (A_i X^{i-1})
        % where A_i = p^i A_0
        for i=1:ninf
            Q = Q - param{3}^i*param{2}*X^(i-1); 
        end       
        % The matrix Q is computed
        
        % Compute |X - F(X)| = |X - (I-Q)/p|, whatever X is
        nF = norm(X-(eye(n,n)-Q)/param{3});
        
        if nargin < 4 
            Bn = 2*Bn-Bn*Q*Bn;
            F = Bn*param{2};
        elseif flag == 0
            % If flag = 0 we have only to initialize the value Bn = B_0
            % Save the matrix B_0 in Bn
            Bn = param{1};    
            F = [];
        elseif flag ~=0
            % If flag ~= 0 we compute only nF = F(X)
            F = [];
        end
             
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXEMPLE 9      Computation of the exponential function exp(A t)
%                Only for Acceleration method.
%                Figure 10 of the paper
%
    case '9'
        
        % F will be the partial sum of the series e^(A*t):
        % I +(*At)/1!+(A*t)^2/2!+...
        
        if nargin < 4 
            ind = ind + 1;              % Increase the counter i
            nterm = nterm*param{1}/ind; % Compute the new term (A^i t^i)/i!
            psum = psum+nterm;          % Add the new term to the previous
                                        %   partial sum
            F = psum;
            nF=norm(X-F);
        elseif flag == 0
            % If flag = 0 we have only to initialize the persistent values
            % ind   counter of the terms
            % psum  partial sum
            % nterm term to be added
            ind = 0;
            psum = eye(n,n);
            nterm = eye(n,n);
            F = [];
            nF = [];
        elseif flag ~=0
            % If flag ~= 0 we compute only nF = F(X)
            nF = norm(X-psum);
            F = [];
        end
               
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXEMPLE 10     Computation of the inverse of (I - A t)
%                Only for Acceleration method.
%                Figure 11 of the paper
%
    case '10'
        
        % F must be the partial sums of (1-A*t)^{-1}
        % I +(A*t)+ (A*t)^2+.......
        
        if nargin < 4 
            nterm = nterm*param{1};     % Compute the new term (A*t)^i
            psum = psum+nterm;          % Add the new term to the previous
                                        %   partial sum
            F = psum;
            nF=norm(X-F);
        elseif flag == 0
            % If flag = 0 we have only to initialize the persistent values
            % psum  partial sum
            % nterm term to be added
            psum = eye(n,n);
            nterm = eye(n,n);
            F = [];
            nF = [];
        elseif flag ~=0
            % If flag ~= 0 we compute only nF = F(X)
            nF = norm(X-psum);
            F = [];
        end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXEMPLE 11     Computation of the square root of A
%                Only for Acceleration method.
%                Figure 12 of the paper
%

    case '11'
        
        if nargin < 4 
            F = X + nterm;
            nF = norm(X-F);
            nterm = -0.5*nterm*(F\nterm); % Compute the new term E_{k+1}
        elseif flag == 0
            % If flag = 0 we have only to initialize the persistent value
            % nterm term 
            nterm = (eye(n,n)-param{1})/2;
            F = [];
            nF = [];
        elseif flag ~=0
            % If flag ~= 0 we cannot compute nF 
            error('We cannot use this example with an unknown solution')
        end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXEMPLE 12     Binomial iteration for the square root of (I-C)
%                Only for Acceleration method.
%                Figure 13 of the paper
%

    case '12'
        % Compute X = 1/2*(C+X^2) 
        F = 0.5*(param{1}+X^2); 
        nF = norm(X-F);
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    otherwise
        error(' Fiter: The example chosen do not exist')
end