function [EPSINI,EPSVEC] = TEA2(EPSINI,EPSVEC,MAXCOL,Y,TOL,IFLAG)
%  TEA2   Second Topological Epsilon-Algorithm 
%                                                                               
%  [EPSINI,EPSVEC] = TEA2(EPSINI,EPSVEC,MAXCOL,Y,TOL,IFLAG)                                    
%                                                                          
%  Arguments:                                                              
%                                                                          
%  EPSINI   Input/output real vector. 
%           In input, before the k-th call of the function, it must contain
%           the new term of the original sequence S_{k-1}.
%           In output, after the k-th call, it contains
%            eps_{k-1}^{(0)}             if k <= MAXCOL+1, k ODD   
%            eps_{k-2}^{(1)}             if k <  MAXCOL+1, k EVEN  
%            eps_{MAXCOL}^{(k-1-MAXCOL)} if k >  MAXCOL+1  
%           Thus, in the successive calls the function returns the sequence
%            eps_0^{(0)}, eps_0^{(1)}, eps_2^{(0)}, eps_2^{(1)} ....
%            eps_MAXCOL^{(0)}, eps_MAXCOL^{(1)}, eps_MAXCOL^{(2)}, .....
%                                                                          
%  EPSVEC   Input/output cell array argument.  
%           It contains after the k-th call the last computed backward
%           diagonal of the epsilon scheme:      
%           - IF k <= MAXCOL+1 
%               eps_0^{(k-1)}, eps_1^{(k-2)}, ... eps_{k-1}^{(0)}
%           - IF k > MAXCOL+1  
%               eps_0^{(k-1)}, eps_1^{(k-2)}, ... eps_MAXCOL^{(k-1-MAXCOL)}
%           Before the first call its input value must be an empty array.
%                                                                          
%  MAXCOL   Input integer giving the index of the last column of the
%           epsilon scheme that the user wants to compute (lower index of
%           eps_m^{(n}). This value must be a positive even integer number
%                                                                          
%  Y        Input real arbitrary vector used for the inner products
%
%  TOL      Input real value used in tests for zero. If abs(X) < TOL, then
%           X is considered to be zero
%                                                                                
%  IFLAG    Input integer to be set to zero at the 'first' call of the
%           function for the initializations of the scheme.  
%           If this value is different from zero, or the argument is
%           missing, the function continues in expanding the previous
%           scheme. For a new application of the algorithm, the user must
%           remember to call the function with IFLAG equal to zero
%
%  Internal working cell array:                                               
%                                                                        
%  WG       Working cell array (3 elements)                                                                                             
%                                                                          
%  REMARKS:                                                                
%                                                                         
%   -  The EPSVEC cell array and the vector Y must not be modified by the 
%      user between two consecutive calls of the function                                    
%   -  To obtain the value eps_{2*j}^{(0)} (2*J <= MAXCOL) the user must
%      call this function (2*J+1) times (including the first call)       
%  

%  AUTHORS: Claude Brezinski and Michela Redivo-Zaglia
%  V1.0 - December 2016
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%  
persistent INDM 
%
%                        ... EXECUTABLE STATEMENTS
%
%                        ... CHECK THE ARGUMENTS
if nargin < 5               % CHECK the number of input arguments
    error(' TEA2 - Not enought input arguments ')
elseif nargout < 2          % CHECK the number of output arguments
    error(' TEA2 - Not enought output arguments ')
elseif (mod(MAXCOL,2) ~= 0) || MAXCOL < 0 % CHECK MAXCOL 
    error(' TEA2 - MAXCOL must be a positive even integer number')
end
%
%                        ... FIRST CALL OF THE FUNCTION WITH IFLAG == 0 
%                            OR TRIVIAL CASE (MAXCOL == 0)
if ((nargin == 6) && (IFLAG == 0)) || (MAXCOL == 0)
%                        ... INITIALIZE THE COUNTER INDM
%       INDM is a counter for the calls of the function. The function
%       at each call computes the INDM+1 backward diagonal (not used
%       in the trivial case, that is when MAXCOL == 0)
    INDM   = 0;
%
%                        ... SET THE BACKWARD DIAGONAL EQUAL TO eps_0^{(0)}
%                            OR TO eps_0^{(n}) (IN THE TRIVIAL CASE)
    EPSVEC{1} = EPSINI;   
return
%
%                        ... END OF THE FIRST CALL OF THE FUNCTION OR
%                            TRIVIAL CASE (MAXCOL == 0)
end
%                        ... NEXT CALLS OF THE FUNCTION
%
%                        ... UPDATE INDM (IT REPRESENTS THE BACKWARD  
%                            DIAGONAL TO BE COMPUTED 
INDM = INDM + 1;
%
%                        ... IF INDM > MAXCOL THEN THE ELEMENT 
%                            eps_MAXCOL^{(NUMBER_OF_CALLS-MAXCOL+1)}
%                            IS COMPUTED
IND = min([INDM,MAXCOL]);
%
%                        ... INITIALIZATION FOR THE AUXILIARY CELL ARRAY
%                            WE SET WG{2} EQUAL TO eps_{0}^(INDM)
WG{2} = EPSINI;
%
%                        ... STARTING LOOP FOR COMPUTING THE ELEMENTS
%                            OF THE ASCENDING DIAGONAL.
%
for INDS = 0:IND - 1
%                        ... IF INDS IS EVEN, COMPUTE A TERM OF AN ODD COLUMN
%
    if (mod(INDS,2) == 0)  
%                        ... COMPUTE THE DENOMINATOR OF THE ODD TERM
%                            AND CHECK IT WITH TOL
        RD = Y'*(WG{2} - EPSVEC{INDS+1});
        if (abs(RD) < TOL) % check for the denominator
            disp(['Value of denominator ',num2str(RD,'%15.5e'), ...
                  ' in odd column ', num2str(INDS+1,'%d')])
            error([' TEA2 - Division by an inner product < TOL.', ... 
                  ' Impossible to continue.'])
        end
%                        ... COMPUTE THE TERM OF THE ODD COLUMN.
        WG{1} = Y / RD;
        if (INDS ~= 0) 
            WG{1} = WG{1} + EPSVEC{INDS}; 
%  
%                        ... SHIFT THE RHOMBUS.
%
                EPSVEC{INDS} = WG{3};
        end 
        WG{3} = WG{2};
        WG{2} = WG{1};
    else
%  
%                        ... IF INDS IS ODD, COMPUTE A TERM OF AN EVEN COLUMN
% 
%                        ... COMPUTE THE DENOMINATOR OF THE EVEN TERM
%                            BY USING THE PREVIOUSLY STORED VECTORS 
%                            DIFFERENCES BETWEEN TO EVEN TERMS, AND CHECK 
%                            IT WITH TOL
%
        RD = (WG{1} - EPSVEC{INDS+1})'*(WG{3} - EPSVEC{INDS});
        if (abs(RD) < TOL) 
            disp(['Value of denominator ',num2str(RD,'%15.5e'), ...
                   ' in even column ', num2str(INDS+1,'%d')])
            error([' TEA2 - Division by an inner product < TOL.', ... 
                  ' Impossible to continue.'])
        end 
%  
%                        ... COMPUTE THE TERM OF THE EVEN COLUMN
        WG{1} = (WG{3} - EPSVEC{INDS})/RD + EPSVEC{INDS};
%  
%                        ... SHIFT THE RHOMBUS
        EPSVEC{INDS} = WG{3}; 
        WG{3} = WG{2};
        WG{2} = WG{1};
    end
%
%                        ... END LOOP FOR INDS
end
%  
%                        ... SAVE FOR THE FINAL VALUES OF THE BACKWORD DIAGONAL
EPSVEC{IND}   = WG{3};  
EPSVEC{IND+1} = WG{2};  
%
%                        ... SET THE RESULT VALUE
if (mod(IND,2) == 0) 
    EPSINI = EPSVEC{IND+1};  
else
    EPSINI = EPSVEC{IND};    
end