classdef  testFormat_array < matlab.unittest.TestCase
    %TESTFORMAT_ARRAY Test method for formatting array
    
    methods(Test)
        
        function testNoInput(testCase)
            p = polysym;
            out = format_array(p);
            testCase.verifyEqual(out,'0');
        end
        
        function testCharInput(testCase)
            p = polysym('s');
            out = format_array(p);
            testCase.verifyEqual(out,'s');
        end
        
        function testCellInput(testCase)
            p = polysym({'s'});
            out = format_array(p);
            testCase.verifyEqual(out,'s');
        end
        
        function testCellVectorInput(testCase)
            p = polysym({'s','t'});
            out = format_array(p);
            testCase.verifyEqual(out,'s t')
        end
        
        function testCellArrayInput(testCase)
            p = polysym({'p','q'; 'r','s'});
            out = format_array(p);
            testCase.verifyEqual(out,['p q'; 'r s'])
        end
        
        function testDoubleInput(testCase)
            p = polysym(pi);
            out = format_array(p);
            testCase.verifyEqual(out,num2str(pi,15));
        end
        
        function testDoubleVectorInput(testCase)
            p = polysym([1 2]);
            out = format_array(p);
            testCase.verifyEqual(out,'1 2')
        end
        
        function testDoubleMatrixInput(testCase)
            p = polysym([1 2; 3 4]);
            out = format_array(p);
            testCase.verifyEqual(out,['1 2';'3 4'])
        end
        
        function testDoubleInputWithPrecision(testCase)
            p = polysym(pi,1);
            out = format_array(p);
            testCase.verifyEqual(out,'3.14159265358979');
        end
        
        function testPolysymInput(testCase)
            p = polysym(polysym('s'));
            out = format_array(p);
            testCase.verifyEqual(out,'s');
        end
        
        function testCharInputWithScalarSize(testCase)
            p = polysym('x',[1 3]);
            out = format_array(p);
            testCase.verifyEqual(out,'x1 x2 x3')
        end
        
        function testCharInputWithVectorSize(testCase)
            p = polysym('x',[2 2]);
            out = format_array(p);
            testCase.verifyEqual(out,['x1_1 x1_2';'x2_1 x2_2'])
        end
        
        function testSquareRootMinusOne(testCase)
            p = polysym(sqrt(-1));
            out = format_array(p);
            testCase.verifyEqual(out,'1*I')
        end
        
        function testImaginaryIntegerMultiplier(testCase)
            p = polysym(5*sqrt(-1));
            out = format_array(p);
            testCase.verifyEqual(out,'5*I')
        end
        
        function testImaginaryIntegerMultiplierVector(testCase)
            p = polysym([3 4]*sqrt(-1));
            out = format_array(p);
            testCase.verifyEqual(out,'3*I 4*I')
        end
        
        function testImaginaryRealMultiplier(testCase)
            r = pi*sqrt(-1);
            p = polysym(r);
            out = format_array(p);
            precision = floor(log10(r/eps(r)));
            testCase.verifyEqual(out,[num2str(pi,precision),'*I'])
        end
        
        function testComplexInput(testCase)
            p = polysym(4+6i);
            out = format_array(p);
            testCase.verifyEqual(out,'4+6*I')
        end
        
        function testDelimiterArg(testCase)
            p = polysym('x',[1 3]);
            out = format_array(p,',');
            testCase.verifyEqual(out,'x1,x2,x3')
        end
        
        function testEmptyDelimiter(testCase)
            p = polysym('x',[1 3]);
            out = format_array(p,[]);
            testCase.verifyEqual(out,'x1 x2 x3')
        end
        
        function testEmptyDelimiterWithJustification(testCase)
            p = polysym('abc ');
            out = format_array(p,[],'right');
            testCase.verifyEqual(out,' abc')
        end
        
        function testNonemptyDelimiterWithJustification(testCase)
            p = polysym({'abc ','x'});
            out = format_array(p,',','right');
            testCase.verifyEqual(out,' abc,x')
        end
        
        function testInvalidJustification(testCase)
            p = polysym({'abc ','x'});
            f = @() format_array(p,',','up');
            testCase.verifyError(f,'MATLAB:strjust:UnknownParameter')
            
        end
    end
end