classdef  testConstructor < matlab.unittest.TestCase
    %TESTCONSTRUCTOR Test polysym constructor
    properties
        d
    end
    
    methods(TestMethodSetup)
        function setup(testCase)
            % Save current vpa precision for restoration later
            testCase.d = digits;
            digits(32)
            
            % No arguments
        end
    end
    
    methods(Test)
        function testNoInput(testCase)
            out = polysym;
            testCase.verifyEqual(out.Value,'0')
            
            % First argument:
            % "X ... can be of class CHAR, POLYSYM, DOUBLE, or SYM ..."
            %----------------------------------------------------------------------
        end
        
        function testFirstArgEmptyChar(testCase)
            out = polysym('');
            testCase.verifyTrue(isempty(out))
        end
        
        function testFirstArgEmptyPolysym(testCase)
            out = polysym(polysym.empty);
            testCase.verifyTrue(isempty(out))
        end
        
        function testFirstArgEmptyDouble(testCase)
            out = polysym([]);
            testCase.verifyTrue(isempty(out))
        end
        
        function testFirstArgEmptyCell(testCase)
            out = polysym({});
            testCase.verifyTrue(isempty(out))
        end
        
        function testFirstArgEmptySym(testCase)
            out = polysym(sym(''));
            testCase.verifyTrue(isempty(out))
        end
        
        function testFirstArgEmptyVpa(testCase)
            out = polysym(vpa([]));
            testCase.verifyTrue(isempty(out))
        end
        
        function testFirstArgEmptyMultipleCols(testCase)
            out = polysym(cell(0,2));
            testCase.verifyEqual(size(out),[0 2])
        end
        
        function testFirstArgCharScalar(testCase)
            out = polysym('s');
            testCase.verifyEqual(out.Value,'s');
        end
        
        function testFirstArgPolysym(testCase)
            out = polysym(polysym('s'));
            testCase.verifyEqual(out.Value,'s');
        end
        
        function testFirstArgDouble(testCase)
            out = polysym(pi);
            testCase.verifyEqual(out.Value,'3.14159265358979');
        end
        
        function testDoubleWithSize(testCase)
            out = polysym(pi,[1 2]);
            testCase.verifyEqual(out.Value,'[3.14159265358979 3.14159265358979]');
        end
        
        function testFirstArgDoubleVector(testCase)
            out = polysym([1 2]);
            testCase.verifyEqual(out(1).Value,'1')
            testCase.verifyEqual(out(2).Value,'2')
        end
        
        function testFirstArgDoubleMatrix(testCase)
            out = polysym([1 2; 3 4]);
            testCase.verifyEqual(out(1,1).Value,'1')
            testCase.verifyEqual(out(1,2).Value,'2')
            testCase.verifyEqual(out(2,1).Value,'3')
            testCase.verifyEqual(out(2,2).Value,'4')
        end
        
        function testFirstArgDoubleMatrixWithSize(testCase)
            out = polysym([1 2; 3 4],[1 2]);
            testCase.verifyEqual(out(1,1).Value,'1')
            testCase.verifyEqual(out(1,2).Value,'2')
            testCase.verifyEqual(out(2,1).Value,'3')
            testCase.verifyEqual(out(2,2).Value,'4')
            testCase.verifyEqual(out(1,3).Value,'1')
            testCase.verifyEqual(out(1,4).Value,'2')
            testCase.verifyEqual(out(2,3).Value,'3')
            testCase.verifyEqual(out(2,4).Value,'4')
            
            % function testFirstArgDouble3D(testCase)
            % f = @() polysym(ones(2,2,2));
            % testCase.verifyError(f,'polysym:polysym:tooManyDimensions')
        end
        
        function testFirstArgSym(testCase)
            out = polysym(sym('x'));
            testCase.verifyEqual(out.Value,'x')
        end
        
        function testFirstArgSymArray(testCase)
            out = polysym(sym('x',2));
            testCase.verifyEqual(out,polysym('x',2))
        end
        
        function testCellArray(testCase)
            str1 = '5419351/1725033';
            str2 = '10838702/1725033';
            out = polysym({str1,str2});
            testCase.verifyEqual(out(1).Value,str1)
            testCase.verifyEqual(out(2).Value,str2)
        end
        
        function testFirstArgVpa(testCase)
            x = vpa(pi);
            out = polysym(x);
            testCase.verifyEqual(out.Value,char(x))
        end
        
        function testFirstArgVpaMoreDigits(testCase)
            % A vpa number is always stored with the number of digits specified by the
            % function DIGITS.
            x = vpa(pi,40);
            y = vpa(pi);
            out = polysym(x);
            testCase.verifyEqual(out.Value,char(y))
            
            % "... or a CELL array of any of these types."
        end
        
        function testCellstrScalar(testCase)
            out = polysym({'s'});
            testCase.verifyEqual(out.Value,'s');
        end
        
        function testCellstrVector(testCase)
            out = polysym({'s','t'});
            testCase.verifyEqual(size(out),[1 2])
            testCase.verifyEqual(out(1).Value,'s')
            testCase.verifyEqual(out(2).Value,'t')
        end
        
        function testCellstrMatrix(testCase)
            out = polysym({'p','q'; 'r','s'});
            testCase.verifyEqual(size(out),[2 2])
            testCase.verifyEqual(out(1,1).Value,'p')
            testCase.verifyEqual(out(1,2).Value,'q')
            testCase.verifyEqual(out(2,1).Value,'r')
            testCase.verifyEqual(out(2,2).Value,'s')
        end
        
        function testCellStringArray(testCase)
            f = @() polysym({['x';'y']});
            testCase.verifyError(f,'polysym:polysym:wrongSize')
        end
        
        function testCellDouble(testCase)
            out = polysym({pi});
            testCase.verifyEqual(out.Value,'3.14159265358979')
        end
        
        function testCellstr(testCase)
            out = polysym({polysym('x')});
            testCase.verifyEqual(out.Value,'x')
        end
        
        function testCellVpa(testCase)
            x = vpa(pi);
            out = polysym({x});
            testCase.verifyEqual(out.Value,upper(char(x)))
            
            % Invalid input should result in an error.
        end
        
        function testInvalidInput(testCase)
            f = @() polysym(int8(1));
            testCase.verifyError(f,'polysym:polysym:invalidInput');
            
            % Char argument:  "X can include formatting instructions as in SPRINTF; the
            % default has the form 'A%d_%d' for a matrix."
        end
        
        function testCharScalarWithFormatting(testCase)
            out = polysym('s%d');
            testCase.verifyEqual(out.Value,'s')
            
            % Second argument:
            % "OBJ=polysym(X,sz), where X is CHAR, also inputs SZ, the size  of the
            % desired array."
            %----------------------------------------------------------------------
        end
        
        function testCharArraySingleArg(testCase)
            f = @() polysym(['s';'t']);
            testCase.verifyError(f,'polysym:polysym:wrongSize')
        end
        
        function testCharInputWithScalarSize(testCase)
            out = polysym('x',2);
            testCase.verifyEqual(size(out),[2 2])
            testCase.verifyEqual(out(1).Value,'x1_1')
            testCase.verifyEqual(out(2).Value,'x2_1')
            testCase.verifyEqual(out(3).Value,'x1_2')
            testCase.verifyEqual(out(4).Value,'x2_2')
        end
        
        function testCharInputRowVector(testCase)
            out = polysym('x',[1 3]);
            testCase.verifyEqual(size(out),[1 3])
            testCase.verifyEqual(out(1).Value,'x1')
            testCase.verifyEqual(out(2).Value,'x2')
            testCase.verifyEqual(out(3).Value,'x3')
        end
        
        function testCharInputColumnVector(testCase)
            out = polysym('x',[3,1]);
            testCase.verifyEqual(size(out),[3 1])
            testCase.verifyEqual(out(1).Value,'x1')
            testCase.verifyEqual(out(2).Value,'x2')
            testCase.verifyEqual(out(3).Value,'x3')
        end
        
        function testCharInputWithVectorSize(testCase)
            out = polysym('x',[2 2]);
            testCase.verifyEqual(size(out),[2 2])
            testCase.verifyEqual(out(1,1).Value,'x1_1')
            testCase.verifyEqual(out(1,2).Value,'x1_2')
            testCase.verifyEqual(out(2,1).Value,'x2_1')
            testCase.verifyEqual(out(2,2).Value,'x2_2')
        end
        
        function testCharEmptyDimensions(testCase)
            f = @() polysym('x',[]);
            testCase.verifyError(f,'MATLAB:polysym:expectedNonempty')
        end
        
        function testCharIndexZero(testCase)
            out = polysym('x',0);
            testCase.verifyTrue(isempty(out))
        end
        
        function testCharIndexZeroVector(testCase)
            out = polysym('x',[1 0]);
            testCase.verifyTrue(isempty(out))
        end
        
        function testCharIndexOne(testCase)
            out = polysym('x',1);
            testCase.verifyEqual(out.Value,'x')
            
            % function testCharTooManyDimensions(testCase)
            % f = @() polysym('x',[2 2 2]);
            % testCase.verifyError(f,'polysym:polysym:tooManyDimensions')
        end
        
        function testCharFormattedRowVector(testCase)
            out = polysym('x%d',[3 1]);
            testCase.verifyEqual(size(out),[3 1])
            testCase.verifyEqual(out(1).Value,'x1')
            testCase.verifyEqual(out(2).Value,'x2')
            testCase.verifyEqual(out(3).Value,'x3')
            
            % "Complex numbers are represented in the form 'X+Y*I', where X and Y  are
            % polynomial expressions and 'I' is reserved for the square root of  -1."
        end
        
        function testSquareRootMinusOne(testCase)
            out = polysym(sqrt(-1));
            testCase.verifyEqual(out.Value,'1*I')
        end
        
        function testImaginaryIntegerMultiplier(testCase)
            out = polysym(5*sqrt(-1));
            testCase.verifyEqual(out.Value,'5*I')
        end
        
        function testImaginaryIntegerMultiplierVector(testCase)
            out = polysym([3 4]*sqrt(-1));
            testCase.verifyEqual(out(1).Value,'3*I')
            testCase.verifyEqual(out(2).Value,'4*I')
        end
        
        function testImaginaryRealMultiplier(testCase)
            out = polysym(pi*1i);
            testCase.verifyEqual(out.Value,'3.14159265358979*I')
        end
        
        function testComplexInput(testCase)
            out = polysym(4+6i);
            testCase.verifyEqual(out.Value,'4+6*I')
        end
        
        function testVpaComplex(testCase)
            x = vpa('sqrt(2)*(1+i)');
            out = polysym(x);
            testCase.verifyEqual(out.Value,upper(char(x)))
        end
    end
    
    methods(TestMethodTeardown)
        function teardown(testCase)
            digits(testCase.d);
        end
    end
end
