classdef test_struct2mat < matlab.unittest.TestCase
    %TEST_STRUCT2MAT Test the method for converting solution structure into matrix
    properties
        input_file_fun = {@(x) x,@(x) make_input_file(x)};
        xsols = {[1 -1],polysym([1 -1]),{1,-1}};
        xsols_struct = {struct('x',[1 -1]), struct('x',polysym([1 -1]))};
    end
    methods(Test)
        function testEmpty(testCase)
            obj = BertiniLab;
            sols = struct.empty;
            f = @() obj.struct2mat(sols);
            testCase.verifyError(f,'BertiniLab:struct2mat:noVariables')
        end
        function testOneVariableRowVectorInput(testCase)
            obj = BertiniLab;
            obj.variable_group = 'x';
            obj.function_def = 'x^2-1';
            for ii=1:numel(testCase.input_file_fun)
                obj = testCase.input_file_fun{ii}(obj);
                for jj=1:numel(testCase.xsols)
                    xs = testCase.xsols{jj};
                    M = obj.struct2mat(xs);
                    testCase.verifyEqual(M,polysym(xs(:).'))
                end
            end
        end
        function testOneVariableRowVectorInputSym(testCase)
            % Test only if the Symbolic Toolbox is installed; otherwise
            % fail by verification
            testCase.assumeNotEmpty(ver('symbolic'))
            obj = BertiniLab;
            obj.variable_group = 'x';
            obj.function_def = 'x^2-1';
            for ii=1:numel(testCase.input_file_fun)
                obj = testCase.input_file_fun{ii}(obj);
                sols = sym([1 -1]);
                M = obj.struct2mat(sols);
                testCase.verifyEqual(M,polysym(sols(:).'))
            end
        end
        function testScalarVariableColumnVectorInput(testCase)
            % A column vector is interpreted as one solution with multiple
            % variables.
            obj = BertiniLab;
            obj.variable_group = 'x';
            obj.function_def = 'x^2-1';
            for ii=1:numel(testCase.input_file_fun)
                obj = testCase.input_file_fun{ii}(obj);
                for jj=1:numel(testCase.xsols)
                    f = @() obj.struct2mat(testCase.xsols{jj}.');
                    testCase.verifyError(f,'BertiniLab:struct2mat:wrongLength')
                end
            end
        end
        function testOneVariableStructInput(testCase)
            obj = BertiniLab;
            obj.variable_group = 'x';
            obj.function_def = 'x^2-1';
            for ii=1:numel(testCase.input_file_fun)
                obj = testCase.input_file_fun{ii}(obj);
                for jj=1:numel(testCase.xsols_struct)
                    xs = testCase.xsols_struct{jj};
                    M = obj.struct2mat(xs);
                    testCase.verifyEqual(M,polysym(xs.x(:).'))
                end
            end
        end
        function testOneVariableInvalidInput(testCase)
            obj = BertiniLab;
            obj.variable_group = 'x';
            obj.function_def = 'x^2-1';
            for ii=1:numel(testCase.input_file_fun)
                obj = testCase.input_file_fun{ii}(obj);
                sols = true;
                f = @() obj.struct2mat(sols);
                testCase.verifyError(f,'MATLAB:polysym:invalidType')
            end
        end
        function testOneVariableNonmatchingFieldNames(testCase)
            obj = BertiniLab;
            obj.variable_group = 'x';
            obj.function_def = 'x^2-1';
            for ii=1:numel(testCase.input_file_fun)
                obj = testCase.input_file_fun{ii}(obj);
                sols = struct('y',[1 -1]);
                f = @() obj.struct2mat(sols);
                testCase.verifyError(f,'BertiniLab:struct2mat:mismatched')
            end
        end
        function testVectorVariable(testCase)
            polysyms mu0 lambda
            v = polysym('v',[1 2]);
            vars = {[mu0 lambda],v};
            A = [1 2; 2 4]; B = [4 -2; -2 1];
            eqns = mu0*A*v.' - lambda*B*v.';
            
            obj = BertiniLab;
            obj.function_def=eqns;
            obj.hom_variable_group=vars;
            for ii=1:numel(testCase.input_file_fun)
                obj = testCase.input_file_fun{ii}(obj);
                
                sols = struct('mu0',randi(100,1),'lambda',randi(100,1),'v',randi(100,1,2));
                actual = obj.struct2mat(sols);
                expected = polysym([sols.mu0; sols.lambda; sols.v.']);
                testCase.verifyEqual(actual,expected)
            end
        end
    end
    
    methods(TestMethodTeardown)
        function teardown(~)
            BertiniClean
        end
    end
end