classdef test_set_get < matlab.unittest.TestCase
    %Test the set/get methods.
    
    properties
        poly_system  % Default system
        poly_solved  % System with defined variables
        xvar = {[],'x',polysym('x'),polysym('v',2)};
        xsols = {[1 -1],polysym([1 -1]),{1,-1}};
        xsols_struct = {struct('x',[1 -1]), struct('x',polysym([1 -1]))};
        data_file_name = {'starting_points','member_points'};
        random_name = {'random','random_real'};
    end
    methods(TestMethodSetup)
        function setup(testCase)
            testCase.poly_system = BertiniLab;
            testCase.poly_solved = BertiniLab('function_def',{'x^2-1'},'variable_group',{'x'});
        end
    end
    
    methods(Test)
        
        % ------------- config --------------
        function testConfig(testCase)
            testCase.poly_system.config = struct('TRACKTYPE',1);
            testCase.verifyTrue(isfield(testCase.poly_system.config,'TRACKTYPE'))
            testCase.verifyEqual(testCase.poly_system.config.TRACKTYPE,1)
        end
        function testConfigNotStruct(testCase)
            f = @() BertiniLab('config','TRACKTYPE');
            testCase.verifyError(f,'MATLAB:set:config:invalidType')
        end
        function testConfigEmpty(testCase)
            f = @() BertiniLab('config',struct.empty);
            testCase.verifyError(f,'MATLAB:set:config:expectedNonempty')
        end
        function testInvalidField(testCase)
            f = @() BertiniLab('var',polysym('x'));
            testCase.verifyError(f,'MATLAB:noPublicFieldForClass')
        end
        
        % ------------- variable --------------
        function testValidVariable(testCase)
            for ii=1:numel(testCase.xvar)
                testCase.poly_system.variable = testCase.xvar{ii};
                testCase.verifyEqual(testCase.poly_system.variable,polysym(testCase.xvar{ii}))
            end
        end
        function testVariableSym(testCase)
            testCase.assumeNotEmpty(ver('symbolic'))
            syms x
            testCase.poly_system.variable = x;
            testCase.verifyEqual(testCase.poly_system.variable,polysym('x'))
        end
        function testVariableInvalidType(testCase)
            f = @() BertiniLab('variable',struct);
            testCase.verifyError(f,'MATLAB:polysym:invalidType')
        end
        function testVariableMultihomogeneous(testCase)
            polysyms('x','y')
            f = @() BertiniLab('variable',{x,y});
            testCase.verifyError(f,'BertiniLab:set:variable:invalidType')
        end
        
        % ------------- variable_group --------------
        function testValidVariableGroup(testCase)
            for ii=1:numel(testCase.xvar)
                testCase.poly_system.variable_group = testCase.xvar{ii};
                testCase.verifyEqual(testCase.poly_system.variable_group,polysym(testCase.xvar{ii}))
            end
        end
        function testVariableGroupSym(testCase)
            testCase.assumeNotEmpty(ver('symbolic'))
            syms x
            testCase.poly_system.variable_group = x;
            testCase.verifyEqual(testCase.poly_system.variable_group,polysym('x'))
        end
        function testVariableGroupInvalidType(testCase)
            f = @() BertiniLab('variable_group',struct);
            testCase.verifyError(f,'MATLAB:polysym:invalidType')
        end
        function testVariableGroupMultihomogeneous(testCase)
            polysyms('x','y')
            testCase.poly_system.variable_group = {x, y};
            testCase.verifyEqual(testCase.poly_system.variable_group,{x,y})
        end
        
        % ------------- hom_variable_group --------------
        function testValidHomVariableGroup(testCase)
            for ii=1:numel(testCase.xvar)
                testCase.poly_system.hom_variable_group = testCase.xvar{ii};
                testCase.verifyEqual(testCase.poly_system.hom_variable_group,polysym(testCase.xvar{ii}))
            end
        end
        function testHomVariableGroupSym(testCase)
            testCase.assumeNotEmpty(ver('symbolic'))
            syms x
            testCase.poly_system.hom_variable_group = x;
            testCase.verifyEqual(testCase.poly_system.hom_variable_group,polysym('x'))
        end
        function testHomVariableGroupInvalidType(testCase)
            f = @() BertiniLab('hom_variable_group',struct);
            testCase.verifyError(f,'MATLAB:polysym:invalidType')
        end
        function testHomVariableGroupMultihomogeneous(testCase)
            polysyms('x','y')
            testCase.poly_system.hom_variable_group = {x, y};
            testCase.verifyEqual(testCase.poly_system.hom_variable_group,{x,y})
        end
        
        % ------------- constant --------------
        function testConstantPolysym(testCase)
            testCase.poly_system.constant = polysym({'x','0'});
            testCase.verifyEqual(testCase.poly_system.constant,polysym({'x','0'}))
        end
        function testConstantSym(testCase)
            testCase.assumeNotEmpty(ver('symbolic'))
            testCase.poly_system.constant = [sym('x') sym(0)];
            testCase.verifyEqual(testCase.poly_system.constant,polysym({'x',char(vpa(0))}))
        end
        function testConstantString(testCase)
            testCase.poly_system.constant = {'x',0};
            testCase.verifyEqual(testCase.poly_system.constant,polysym({'x','0'}))
        end
        function testConstantWrongSize(testCase)
            f = @() BertiniLab('constant',polysym('x'));
            testCase.verifyError(f,'MATLAB:set:constant:incorrectNumcols')
        end
        function testNonEmptyThenEmpty(testCase)
            testCase.poly_system.constant = {'x',0};
            testCase.poly_system.constant = [];
            testCase.verifyEqual(testCase.poly_system.constant,polysym.empty(0,2))
        end
        
        % ------------- function_def --------------
        function testFunctionDefsOneColumn(testCase)
            fnames = polysym('eqn',[4 1]);
            fvals = polysym('x',[4 1]);
            testCase.poly_system.function_def = fvals;
            testCase.verifyEqual(testCase.poly_system.function_def,fvals)
            testCase.verifyEqual(testCase.poly_system.function_name,fnames)
        end
        function testFunctionDefsWithName(testCase)
            testCase.poly_system.function_name = polysym('x');
            testCase.poly_system.function_def = polysym('0');
            testCase.verifyEqual(testCase.poly_system.function_name,polysym('x'))
            testCase.verifyEqual(testCase.poly_system.function_def,polysym('0'))
        end
        function testFunctionDefsInvalidType(testCase)
            f = @() BertiniLab('function_def',struct);
            testCase.verifyError(f,'MATLAB:polysym:invalidType')
        end
        
        %-------------- Random --------------------
        function testRandomPolysym(testCase)
            polysyms x
            for ii=1:numel(testCase.random_name)
                testCase.poly_system.(testCase.random_name{ii}) = x;
                testCase.verifyEqual(testCase.poly_system.(testCase.random_name{ii}),x)
            end
        end
        function testRandomSym(testCase)
            testCase.assumeNotEmpty(ver('symbolic'))
            syms x
            for ii=1:numel(testCase.random_name)
                testCase.poly_system.(testCase.random_name{ii}) = x;
                testCase.verifyEqual(testCase.poly_system.(testCase.random_name{ii}),polysym('x'))
            end
        end
        
        function testRandomString(testCase)
            for ii=1:numel(testCase.random_name)
                testCase.poly_system.(testCase.random_name{ii}) = 'x';
                testCase.verifyEqual(testCase.poly_system.(testCase.random_name{ii}),polysym('x'))
            end
        end
        
        function testRandomMatrix(testCase)
            for ii=1:numel(testCase.random_name)
                f = @() BertiniLab(testCase.random_name{ii},polysym('v',2));
                testCase.verifyError(f,['MATLAB:set:',testCase.random_name{ii},':expectedVector'])
            end
        end
        
        function testRandomEmpty(testCase)
            for ii=1:numel(testCase.random_name)
                f = @() BertiniLab(testCase.random_name{ii},[]);
                testCase.verifyError(f,['MATLAB:set:',testCase.random_name{ii},':expectedVector'])
            end
        end
        
        function testRandomInvalidType(testCase)
            for ii=1:numel(testCase.random_name)
                f = @() BertiniLab(testCase.random_name{ii},struct);
                testCase.verifyError(f,'MATLAB:polysym:invalidType')
            end
        end
        
        %-------------- definedSubfunction --------------------
        
        function testDefinedSubfunctionPolysym(testCase)
            polysyms x
            testCase.poly_system.definedSubfunction = x;
            testCase.verifyEqual(testCase.poly_system.definedSubfunction,x)
        end
        
        function testDefinedSubfunctionSym(testCase)
            testCase.assumeNotEmpty(ver('symbolic'))
            syms x
            testCase.poly_system.definedSubfunction = x;
            testCase.verifyEqual(testCase.poly_system.definedSubfunction,polysym('x'))
        end
        
        function testDefinedSubfunctionString(testCase)
            testCase.poly_system.definedSubfunction = 'x';
            testCase.verifyEqual(testCase.poly_system.definedSubfunction,polysym('x'))
        end
        
        function testDefinedSubfunctionVector(testCase)
            f = @() BertiniLab('definedSubfunction',polysym('x',[1 2]));
            testCase.verifyError(f,'MATLAB:set:definedSubfunction:expectedScalar')
        end
        
        function testDefinedSubfunctionEmpty(testCase)
            f = @() BertiniLab('definedSubfunction',[]);
            testCase.verifyError(f,'MATLAB:set:definedSubfunction:expectedScalar')
        end
        
        function testDefinedSubfunctionInvalidType(testCase)
            f = @() BertiniLab('definedSubfunction',struct);
            testCase.verifyError(f,'MATLAB:polysym:invalidType')
        end
        
        function testDefinedSubfunctionNoFile(testCase)
            f = @() BertiniLab('definedSubfunction',struct);
            testCase.verifyError(f,'MATLAB:polysym:invalidType')
        end
        
        % ------------- starting_points, member_points --------------
        function testData_pointsSym(testCase)
            testCase.assumeNotEmpty(ver('symbolic'))
            syms x
            for jj=1:numel(testCase.data_file_name)
                testCase.poly_solved.(testCase.data_file_name{jj}) = x;
                testCase.verifyEqual(testCase.poly_solved.(testCase.data_file_name{jj}),polysym('x'))
            end
        end
        function testData_pointsStruct(testCase)
            for ii=1:numel(testCase.xsols_struct)
                for jj=1:numel(testCase.data_file_name)
                    testCase.poly_solved.(testCase.data_file_name{jj}) = testCase.xsols_struct{ii};
                    testCase.verifyEqual(testCase.poly_solved.(testCase.data_file_name{jj}),polysym(testCase.xsols_struct{ii}.x(:).'))
                end
            end
        end
        function testData_pointsOtherValid(testCase)
            for ii=1:numel(testCase.xsols)
                for jj=1:numel(testCase.data_file_name)
                    xs = testCase.xsols{ii};
                    testCase.poly_solved.(testCase.data_file_name{jj}) = testCase.xsols{ii};
                    testCase.verifyEqual(testCase.poly_solved.(testCase.data_file_name{jj}),polysym(xs(:).'))
                end
            end
        end
        function testData_pointsMultihomogeneous(testCase)
            polysyms('x','y')
            for jj=1:numel(testCase.data_file_name)
                testCase.poly_solved.(testCase.data_file_name{jj})={0,1};
                testCase.verifyEqual(testCase.poly_solved.(testCase.data_file_name{jj}),polysym(0:1))
            end
        end
        
        % ------------- final_parameters --------------
        function testFinal_parametersSym(testCase)
            testCase.assumeNotEmpty(ver('symbolic'))
            syms x
            testCase.poly_solved.final_parameters = x;
            testCase.verifyEqual(testCase.poly_solved.final_parameters,polysym('x'))
        end
        function testFinal_parametersStruct(testCase)
            for ii=1:numel(testCase.xsols_struct)
                f = @() BertiniLab('variable_group','x','final_parameters',testCase.xsols_struct{ii});
                testCase.verifyError(f,'MATLAB:polysym:invalidType')
            end
        end
        function testFinal_parametersValidTypes(testCase)
            for ii=1:numel(testCase.xsols)
                xs = testCase.xsols{ii};
                testCase.poly_solved.final_parameters = xs;
                testCase.verifyEqual(testCase.poly_solved.final_parameters,polysym(xs(:).'))
            end
        end
        function testFinal_parametersMultihomogeneous(testCase)
            polysyms('x','y')
            testCase.poly_solved.final_parameters={0,1};
            testCase.verifyEqual(testCase.poly_solved.final_parameters,polysym(0:1))
        end
        
        % ------------- Output_file_names --------------
        function testNoSummary(testCase)
            testCase.verifyTrue(isempty(testCase.poly_system.output_file_names))
        end
        
        function testSummary1(testCase)
            ps = BertiniLab('function_def',{'x^2-1'; 'y^2-4'},'variable_group',{'x','y'});
            ps = solve(ps);
            fnames = {'main_data','raw_solutions','raw_data','real_finite_solutions', ...
                'finite_solutions','nonsingular_solutions','singular_solutions'};
            testCase.verifyEqual(ps.output_file_names,fnames)
        end
        
        % ------------- Protected properties --------------
        function testTryToSetSolveSummary(testCase)
            f = @() eval('testCase.poly_system.solve_summary=rand');
            testCase.verifyError(f,'MATLAB:class:SetProhibited')
        end
        function testEchoFlag(testCase)
            ps = testCase.poly_system;
            ps.view_progress = true;
            testCase.verifyEqual(ps.echo_flag,'-echo')
        end
    end
    
    methods(TestMethodTeardown)
        function teardown(~)
            BertiniClean
        end
    end
end

