classdef test_read_solutions < matlab.unittest.TestCase
    %Test the method for reading solutions
    
    properties
        poly_system
    end
    methods(TestMethodSetup)
        function nameFiles(testCase)
            BertiniClean
            
            polysyms('x','y')
            obj = BertiniLab('function_def',[x^2-1; x+y-1],'variable_group',[x y]);
            testCase.poly_system = solve(obj);
        end
    end
    
    methods(Test)
        function testNoArguments(testCase)
            f = @() testCase.poly_system.read_solutions;
            testCase.verifyError(f,'MATLAB:minrhs')
        end
        
        function testNoDataFile(testCase)
            delete('raw_data')
            f = @() testCase.poly_system.read_solutions('raw_data');
            testCase.verifyError(f,'BertiniLab:read_solutions:openError')
        end
        
        function testEmptyFile(testCase)
            data = testCase.poly_system.read_solutions('failed_paths');
            testCase.verifyTrue(isempty(data))
        end
        
        function testNoSolutions(testCase)
            data = testCase.poly_system.read_solutions('singular_solutions');
            testCase.verifyTrue(isempty(data))
        end
        
        function testNonsingularSolutions(testCase)
            fid = fopen('nonsingular_solutions','w');
            fprintf(fid,'1\n\n');
            x = [1 2; 3 4];
            fprintf(fid,'%17.16f %17.16f\n',x.');
            fclose(fid);
            data = testCase.poly_system.read_solutions('nonsingular_solutions');
            testCase.verifyEqual(data,polysym({'1.0000000000000000+2.0000000000000000*I'; ...
                '3.0000000000000000+4.0000000000000000*I'}))
        end
        
        function testNoMainData(testCase)
            delete('main_data')
            fid = fopen('nonsingular_solutions','w');
            fprintf(fid,'1\n\n');
            x = [1 2; 3 4];
            fprintf(fid,'%17.16f %17.16f\n',x.');
            fclose(fid);
            data = testCase.poly_system.read_solutions('nonsingular_solutions');
            testCase.verifyEqual(data,polysym({'1.0000000000000000+2.0000000000000000*I'; ...
                '3.0000000000000000+4.0000000000000000*I'}))
        end
        
        function testRawSolutions(testCase)
            % Special case because raw_solutions has path numbers
            fid = fopen('raw_solutions','w');
            fprintf(fid,'1\n\n0\n');
            x = [1 2; 3 4];
            fprintf(fid,'%17.16f %17.16f\n',x.');
            fclose(fid);
            data = testCase.poly_system.read_solutions('raw_solutions');
            testCase.verifyEqual(data,polysym({'1.0000000000000000+2.0000000000000000*I'; ...
                '3.0000000000000000+4.0000000000000000*I'}))
        end
        
        function testRawSolutionsWrongPathNumber(testCase)
            fid = fopen('raw_solutions','w');
            fprintf(fid,'1\n\n1\n'); % should be path 0
            x = rand(2);
            fprintf(fid,'%17.16f %17.16f\n',x.');
            fclose(fid);
            f = @() testCase.poly_system.read_solutions('raw_solutions');
            testCase.verifyError(f,'BertiniLab:read_solutions:wrongNumber')
        end
        
        function testMultihomogeneous(testCase)
            % Make sure number of variables is correct in multihomogenous
            % case
            polysyms mu0 lambda; % Use 'mu0' because 'mu' is a Matlab toolbox command
            v = polysym('v',[1 2]);
            fdef = polysym('f',[1 2]);
            A = [1 2; 2 4]; B = [4 -2; -2 1];
            fval = mu0*A*v.' - lambda*B*v.';
            poly_multihomo = BertiniLab('function_def',fdef(:),'function_def',fval(:), ...
                'hom_variable_group',{[mu0 lambda],v});
            poly_multihomo = solve(poly_multihomo);
            
            fid = fopen('nonsingular_solutions','w');
            fprintf(fid,'1\n\n');
            x = [1 3 5 7; 2 3 6 8].';
            fprintf(fid,'%17.16f %17.16f\n',x.');
            fclose(fid);
            data = poly_multihomo.read_solutions('nonsingular_solutions');
            testCase.verifyEqual(data,polysym({'1.0000000000000000+2.0000000000000000*I'; ...
                '3.0000000000000000+3.0000000000000000*I'; ...
                '5.0000000000000000+6.0000000000000000*I'; ...
                '7.0000000000000000+8.0000000000000000*I'}))
        end
    end
    
    methods(TestMethodTeardown)
        function teardown(~)
            BertiniClean
        end
    end
end