classdef test_make_input_file < matlab.unittest.TestCase
%TEST_MAKE_INPUT_FILE Test the method for making input files

    properties
        default_input_name
        user_input_name
    end
    methods(TestMethodSetup)
        function setup(testCase)
            testCase.default_input_name = 'input';
            testCase.user_input_name = 'my_input';
            
            % Delete the input file, if it exists
            warning('off','MATLAB:DELETE:FileNotFound')
            delete(testCase.default_input_name,testCase.user_input_name);
            warning('on','MATLAB:DELETE:FileNotFound')
        end
    end
    
    methods(Test)
        function testNoFunction(testCase)
            obj = BertiniLab;
            f = @() make_input_file(obj);
            testCase.verifyError(f,'BertiniLab:make_input_file:noFunction')
            testCase.verifyEqual(exist(fullfile(pwd,testCase.default_input_name),'file'),0)
        end
        
        function testNoDeclarations(testCase)
            obj = BertiniLab;
            obj.function_def = polysym({'f','x'});
            f = @() make_input_file(obj);
            testCase.verifyError(f,'BertiniLab:make_declarations:noVariables')
        end
        
        function testMinimal(testCase)
            obj = BertiniLab;
            obj.function_def = polysym({'x'});
            obj.variable_group = polysym('x');
            make_input_file(obj);
            fid = fopen(fullfile(pwd,testCase.default_input_name));
            C = textscan(fid,'%s');
            testCase.verifyEqual(C{1},{'CONFIG'; 'END;'; 'INPUT'; 'variable_group'; 'x;'; ...
                'function'; 'eqn;'; 'eqn=x;'; 'END;'})
        end
        
        function testUserFileName(testCase)
            obj = BertiniLab;
            obj.function_def = polysym({'x'});
            obj.variable_group = polysym('x');
            make_input_file(obj,testCase.user_input_name);
            fid = fopen(fullfile(pwd,testCase.user_input_name));
            C = textscan(fid,'%s');
            testCase.verifyEqual(C{1},{'CONFIG'; 'END;'; 'INPUT'; 'variable_group'; 'x;'; ...
                'function'; 'eqn;'; 'eqn=x;'; 'END;'})
        end
        
        function testMultiHomogeneous(testCase)
            obj = BertiniLab;
            polysyms('x','y')
            obj.variable_group = {x,y};
            obj.function_def = x*y;
            make_input_file(obj);
            fid = fopen(fullfile(pwd,testCase.default_input_name));
            C = textscan(fid,'%s');
            testCase.verifyEqual(C{1},{'CONFIG'; 'END;'; 'INPUT'; 'variable_group'; 'x;'; ...
                'variable_group'; 'y;'; 'function'; 'eqn;'; 'eqn=x*y;'; 'END;'})
        end
        
        function testMinimalWithFunctionName(testCase)
            obj = BertiniLab;
            obj.function_def = polysym('x');
            obj.function_name = polysym('f');
            obj.variable_group = polysym('x');
            make_input_file(obj);
            fid = fopen(fullfile(pwd,testCase.default_input_name));
            C = textscan(fid,'%s');
            testCase.verifyEqual(C{1},{'CONFIG'; 'END;'; 'INPUT'; 'variable_group'; 'x;'; ...
                'function'; 'f;'; 'f=x;'; 'END;'})
        end
        
        function testWithConfig(testCase)
            obj = BertiniLab;
            obj.function_def = polysym('x');
            obj.function_name = polysym('f');
            obj.variable_group = polysym('x');
            obj.config = struct('TRACKTYPE',1);
            make_input_file(obj);
            fid = fopen(fullfile(pwd,testCase.default_input_name));
            C = textscan(fid,'%s');
            testCase.verifyEqual(C{1},{'CONFIG'; 'TRACKTYPE:'; '1;'; 'END;'; 'INPUT'; 'variable_group'; 'x;'; ...
                'function'; 'f;'; 'f=x;'; 'END;'})
        end
        
        function testWithAssignment(testCase)
            obj = BertiniLab;
            obj.function_name = polysym('f');
            obj.function_def = polysym('x+C');
            obj.variable_group = polysym('x');
            obj.constant = polysym({'C','Pi'});
            make_input_file(obj);
            fid = fopen(fullfile(pwd,testCase.default_input_name));
            C = textscan(fid,'%s');
            testCase.verifyEqual(C{1},{'CONFIG'; 'END;'; 'INPUT'; 'variable_group'; 'x;'; ...
                'constant'; 'C;'; 'function'; 'f;'; 'C=Pi;'; 'f=x+C;'; 'END;'})
        end
        
        function testLCpi(testCase)
            obj = BertiniLab;
            obj.function_name = polysym('f');
            obj.function_def = polysym('x+C');
            obj.variable_group = polysym('x');
            obj.constant = polysym({'C','pi'});
            make_input_file(obj);
            fid = fopen(fullfile(pwd,testCase.default_input_name));
            C = textscan(fid,'%s');
            testCase.verifyEqual(C{1},{'CONFIG'; 'END;'; 'INPUT'; 'variable_group'; 'x;'; ...
                'constant'; 'C;'; 'function'; 'f;'; 'C=Pi;'; 'f=x+C;'; 'END;'})
        end
        
        function testUserHomotopyNoVariables(testCase)
            obj = BertiniLab;
            obj.config = struct('USERHOMOTOPY',1);
            obj.function_def = polysym({'f','x'});
            f = @() make_input_file(obj);
            testCase.verifyError(f,'BertiniLab:make_declarations:noVariables')
        end
        
        function testUserHomotopy(testCase)
            obj = BertiniLab;
            obj.config = struct('USERHOMOTOPY',1);
            x = polysym('x'); t = polysym('t'); s = polysym('s'); gamma = polysym('gamma');
            obj.function_name = polysym('f');
            obj.function_def = polysym((x^2-1)*gamma*s + (x^2)*(1-s));
            obj.variable = x;
            obj.pathvariable = t;
            obj.parameter = [s t];
            obj.constant = [gamma polysym(0.8-1.2*1i)];
            make_input_file(obj);
            fid = fopen(fullfile(pwd,testCase.default_input_name));
            C = textscan(fid,'%s');
            testCase.verifyEqual(C{1},{'CONFIG'; 'USERHOMOTOPY:'; '1;'; 'END;'; 'INPUT'; 'variable'; 'x;'; ...
                'pathvariable'; 't;'; 'constant'; 'gamma;'; 'parameter'; 's;'; 'function'; 'f;'; 'gamma=0.8-1.2*I;'; ...
                's=t;';'f=(x^2-1)*gamma*s+x^2*(1-s);'; 'END;'})
        end
        
        function testParameterHomotopy(testCase)
            config = struct('ParameterHomotopy',1);
            polysyms a0 a1 a2 a3 a4 a5 a6 x
            f = a0+x*(a1+x*(a2+x*(a3+x*(a4+x*(a5+x*a6)))));
            obj = BertiniLab('function_def',f,'variable_group',x, ...
                'parameter',[a0 a1 a2 a3 a4 a5 a6],'config',config);
            make_input_file(obj);
            fid = fopen(fullfile(pwd,testCase.default_input_name));
            C = textscan(fid,'%s');
            testCase.verifyEqual(C{1},{'CONFIG'; 'ParameterHomotopy:'; '1;'; 'END;'; 'INPUT'; ...
                'variable_group'; 'x;'; 'parameter'; 'a0,'; 'a1,'; 'a2,'; 'a3,'; ...
                'a4,'; 'a5,'; 'a6;'; 'function'; 'eqn;'; ...
                'eqn=a0+x*(a1+x*(a2+x*(a3+x*(a4+x*(a5+x*a6)))));'; 'END;'})
        end
    end
    
    methods(TestMethodTeardown)
        function teardown(testCase)
            warning('off','MATLAB:DELETE:FileNotFound')
            delete(testCase.default_input_name,testCase.user_input_name);
            warning('on','MATLAB:DELETE:FileNotFound')
        end
    end
end