classdef test_evaluate < matlab.unittest.TestCase
    %Test the evaluate method.
    
    properties
        alt_input_name
        regular_system
        user_homotopy
        parameter_homotopy
    end
    
    methods(TestMethodSetup)
        function setup(testCase)
            BertiniClean
            
            testCase.alt_input_name = 'alt_input';
            if exist(testCase.alt_input_name,'file')==2
                delete(testCase.alt_input_name)
            end
            
            config = struct('SharpenDigits',20);
            polysyms x y
            circle = x^2 + (y-1)^2 - 1;
            parabola = y-2*x^2;
            testCase.regular_system = BertiniLab('function_def',[circle; parabola], 'variable_group',[x y], ...
                'config',config);
            
            polysyms z H Pi t s
            config = struct('UserHomotopy',1); % User-defined homotopy
            q = polysym('q',[2 1]);
            qvals = [cos(2*Pi*(1-t)); sin(2*Pi*(1-t))];
            testCase.user_homotopy = BertiniLab('config',config,'function_def',z^2-s,'function_name',H, ...
                'variable',z,'pathvariable',t,'parameter',[q qvals],'subfunction', ...
                [s q(1)+1i*q(2)],'starting_points',1);
            
            polysyms x
            p = polysym('a',[1 7]);
            config = struct('ParameterHomotopy',1);
            f = polyval(p,x);
            testCase.parameter_homotopy = BertiniLab('function_def',f,'variable_group',x, ...
                'parameter',p,'config',config);
        end
    end
    
    methods(Test)
        
        % Regular homotopy
        function testRegularEmptyPoint(testCase)
            f = @() testCase.regular_system.evaluate;
            testCase.verifyError(f,'BertiniLab:evaluate:minrhs')
        end
        function testRegularNoPoints(testCase)
            f = @() testCase.regular_system.evaluate([]);
            testCase.verifyError(f,'BertiniLab:struct2mat:wrongLength')
        end
        function testRegularValueOnly(testCase)
            fvals = testCase.regular_system.evaluate([.75, 2; 1e-15 .25].');
            testCase.verifyEqual(double(fvals),[0.5625 -0.4375; 0.8750 0.25],'AbsTol',eps*10)
        end
        function testRegularValueAndJacobian(testCase)
            [fvals,J] = testCase.regular_system.evaluate([.75 2; 1e-15 .25].');
            testCase.verifyEqual(double(fvals),[0.5625 -0.4375; 0.8750 0.25],'AbsTol',eps*10)
            testCase.verifySize(J,[1 2])
            testCase.verifyEqual(double(J{1}),[1.5 2; -3 1],'AbsTol',eps*10)
            testCase.verifyEqual(double(J{2}),[2e-15 -1.5; -4e-15 1],'AbsTol',eps*10)
        end
        function testRegularChangeInputFileName(testCase)
            fvals = testCase.regular_system.evaluate([.75, 2; 1e-15 .25].',testCase.alt_input_name);
            testCase.verifyEqual(exist(testCase.alt_input_name,'file'),2)
            testCase.verifyEqual(double(fvals),[0.5625 -0.4375; 0.8750 0.25],'AbsTol',eps*10)
        end
        
        % User homotopy
        function testUserEmptyPoint(testCase)
            f = @() testCase.user_homotopy.evaluate;
            testCase.verifyError(f,'BertiniLab:evaluate:minrhs')
        end
        function testUserNoPoints(testCase)
            f = @() testCase.user_homotopy.evaluate([]);
            testCase.verifyError(f,'BertiniLab:struct2mat:wrongLength')
        end
        function testUserValueOnly(testCase)
            fvals = testCase.user_homotopy.evaluate(1,1);
            testCase.verifyEqual(exist(fullfile(pwd,'start_time'),'file'),2)
            testCase.verifyEqual(double(fvals),0,'AbsTol',eps*10)
        end
        function testUserValueAndJacobian(testCase)
            [fvals,J] = testCase.user_homotopy.evaluate(1,1);
            testCase.verifyEqual(exist(fullfile(pwd,'start_time'),'file'),2)
            testCase.verifyEqual(double(fvals),0,'AbsTol',eps*10)
            testCase.verifyEqual(double(J{1}),2,'AbsTol',eps*10)
        end
        function testUserNoParam(testCase)
            fvals = testCase.user_homotopy.evaluate(1);
            testCase.verifyEqual(exist(fullfile(pwd,'start_time'),'file'),0)
            testCase.verifyEqual(double(fvals),0,'AbsTol',eps*10)
        end
        function testUserEmptyParam(testCase)
            fvals = testCase.user_homotopy.evaluate(1,[]);
            testCase.verifyEqual(exist(fullfile(pwd,'start_time'),'file'),0)
            testCase.verifyEqual(double(fvals),0,'AbsTol',eps*10)
        end
        function testUserParamWrongSize(testCase)
            % Extra parameters are ignored
            fvals = testCase.user_homotopy.evaluate(1,[1 2]);
            testCase.verifyEqual(exist(fullfile(pwd,'start_time'),'file'),2)
            testCase.verifyEqual(double(fvals),0,'AbsTol',eps*10)
        end
        function testUserChangeInputFileName(testCase)
            fvals = testCase.user_homotopy.evaluate(1,1,testCase.alt_input_name);
            testCase.verifyEqual(exist(fullfile(pwd,'start_time'),'file'),2)
            testCase.verifyEqual(exist(testCase.alt_input_name,'file'),2)
            testCase.verifyEqual(double(fvals),0,'AbsTol',eps*10)
        end
        
        
        % Parameter homotopy
        function testParameterEmptyPoint(testCase)
            f = @() testCase.parameter_homotopy.evaluate;
            testCase.verifyError(f,'BertiniLab:evaluate:minrhs')
        end
        function testParameterNoPoints(testCase)
            f = @() testCase.parameter_homotopy.evaluate([]);
            testCase.verifyError(f,'BertiniLab:struct2mat:wrongLength')
        end
        function testWrongNumberOfParameters(testCase)
            poly_sys = testCase.parameter_homotopy;
            poly_sys = poly_sys.solve;
            poly_sys.config.ParameterHomotopy=2;
            f = @() poly_sys.evaluate(1,1);
            testCase.verifyError(f,'BertiniLab:solve:callFailed')
        end
        function testParameterValueOnly(testCase)
            p = poly(1:6).';
            x = -1;
            poly_sys = testCase.parameter_homotopy;
            poly_sys = poly_sys.solve;
            poly_sys.config.ParameterHomotopy=2;
            fvals = poly_sys.evaluate(x,p);
            testCase.verifyEqual(exist(fullfile(pwd,'start'),'file'),2)
            testCase.verifyEqual(exist(fullfile(pwd,'start_parameters'),'file'),2)
            testCase.verifyEqual(exist(fullfile(pwd,'final_parameters'),'file'),2)
            testCase.verifyEqual(double(fvals),polyval(p,x))
        end
        function testParameterValueAndJacobian(testCase)
            p = poly(1:6).';
            x = -1;
            poly_sys = testCase.parameter_homotopy;
            poly_sys = poly_sys.solve;
            poly_sys.config.ParameterHomotopy=2;
            [fvals,J,Jp] = poly_sys.evaluate(x,p);
            testCase.verifyEqual(exist(fullfile(pwd,'start'),'file'),2)
            testCase.verifyEqual(exist(fullfile(pwd,'start_parameters'),'file'),2)
            testCase.verifyEqual(exist(fullfile(pwd,'final_parameters'),'file'),2)
            testCase.verifyEqual(double(fvals),polyval(p,x))
            testCase.verifyEqual(double(fvals),polyval(p,x))
            testCase.verifyEqual(double(J{1}),polyval(polyder(p),x))
            testCase.verifyEqual(double(Jp{1}),(-1).^(0:6))
        end
        function testParameterEmptyParam(testCase)
            poly_sys = testCase.parameter_homotopy;
            poly_sys = poly_sys.solve;
            poly_sys.config.ParameterHomotopy=2;
            f = @() poly_sys.evaluate(1,[]);
            testCase.verifyError(f,'BertiniLab:solve:callFailed')
        end
        function testParameterChangeInputFileName(testCase)
            p = poly(1:6).';
            x = -1;
            poly_sys = testCase.parameter_homotopy;
            poly_sys = poly_sys.solve;
            poly_sys.config.ParameterHomotopy=2;
            movefile(fullfile(pwd,'input'),fullfile(pwd,testCase.alt_input_name))
            fvals = poly_sys.evaluate(x,p,testCase.alt_input_name);
            testCase.verifyEqual(exist(fullfile(pwd,'input'),'file'),0)
            testCase.verifyEqual(exist(testCase.alt_input_name,'file'),2)
            testCase.verifyEqual(double(fvals),polyval(p,x))
        end
        
    end
    
    methods(TestMethodTeardown)
        function teardown(testCase)
            BertiniClean
            if exist(testCase.alt_input_name,'file')==2
                delete(testCase.alt_input_name)
            end
        end
    end
end

