classdef test_dehomogenize < matlab.unittest.TestCase
    %Test the method for dehomogenizing data
    
    properties
        vars
        poly_system
    end
    methods(TestMethodSetup)
        function nameFiles(testCase)
            BertiniClean
            
            polysyms('x','y')
            testCase.vars = [x y];
            testCase.poly_system = BertiniLab('function_def',[x^2-1; x+y-1]);
        end
    end
    
    methods(Test)
        function testNoArguments(testCase)
            f = @() testCase.poly_system.dehomogenize;
            testCase.verifyError(f,'MATLAB:minrhs')
        end
        
        function testNoVarsNoData(testCase)
            actual = testCase.poly_system.dehomogenize([]);
            testCase.verifyEmpty(actual)
        end
        
        function testOneGroupNoData(testCase)
            p = testCase.poly_system;
            p.variable_group = testCase.vars;
            p = p.make_input_file;
            f = @() p.dehomogenize([]);
            testCase.verifyError(f,'BertiniLab:dehomogenize:wrongSize')
        end
        
        function testOneGroupAffineData(testCase)
            p = testCase.poly_system;
            p.variable_group = testCase.vars;
            p = p.make_input_file;
            expected = randi(10,size(testCase.vars)).';
            actual = p.dehomogenize(expected);
            testCase.verifyEqual(actual,expected)
        end
        
        function testRowVector(testCase)
            p = testCase.poly_system;
            p.variable_group = testCase.vars;
            p = p.make_input_file;
            expected = [1 2];
            actual = p.dehomogenize(expected);
            testCase.verifyEqual(actual.',expected)
        end
        
        function testOneGroupHomogenizedData(testCase)
            p = testCase.poly_system;
            p.variable_group = testCase.vars;
            p = p.make_input_file;
            x = randi(10,length(testCase.vars)+1,1);
            actual = p.dehomogenize(x);
            expected = x(2:end)/x(1);
            testCase.verifyEqual(actual,expected,'AbsTol',10*eps)
        end
        
        function testTwoGroupsAffineData(testCase)
            p = testCase.poly_system;
            p.variable_group = num2cell(testCase.vars);
            p = p.make_input_file;
            expected = randi(10,size(testCase.vars)).';
            actual = p.dehomogenize(expected);
            testCase.verifyEqual(actual,expected)
        end
        
        function testTwoGroupsHomogenizedData(testCase)
            p = testCase.poly_system;
            p.variable_group = num2cell(testCase.vars);
            p = p.make_input_file;
            x = randi(10,length(testCase.vars)+2,1);
            actual = p.dehomogenize(x);
            expected = [x(3)/x(1); x(4)/x(2)];
            testCase.verifyEqual(actual,expected,'AbsTol',10*eps)
        end
        
        function testMixed(testCase)
            % Mixed homogeneous and affine
            polysyms x y z
            p = BertiniLab('function_def',[x*z+y; y*z+x], ...
                'hom_variable_group',[x y],'variable_group',z); %#ok<NODEF>
            p = p.make_input_file;
            x = randi(10,4,1);
            actual = p.dehomogenize(x);
            expected = [x(2)/x(1); x(3); x(4)];
            testCase.verifyEqual(actual,expected,'AbsTol',10*eps)   
        end
    end
    
    methods(TestMethodTeardown)
        function teardown(~)
            BertiniClean
        end
    end
end