%% Sharpening using Newton's method
% Section 7.1 from _Numerically solving polynomial systems with Bertini_,
% by Daniel J. Bates, Jonathan D. Haunstein, and Andrew J. Sommese and
% Charles W. Wampler (SIAM 2013).
%
% Sharpen the solutions to the system
%
% $$ x^2+(y-1)^2=1, \quad y=2x^2, $$
%
% which is the intersection of a circle and a parabola.
clear all

config = struct('SharpenDigits',20);
polysyms x y
circle = x^2 + (y-1)^2 - 1;
parabola = y-2*x^2;

circle_parabola_intersection = BertiniLab('function_def',[circle; parabola], 'variable_group',[x y], ...
    'config',config);
circle_parabola_intersection = solve(circle_parabola_intersection);

%%
% This has two nonsingular solutions $(\pm \sqrt{3}/2, 3/2)$:
sols = circle_parabola_intersection.match_solutions('nonsingular_solutions');
xsols = double(sols.x); ysols = double(sols.y);
fprintf('%12s  %34s\n','x','y')
fprintf('%24.20g %24.20g\n',real([xsols ysols]).')

%%
% It also has the singular solution $(0,0)$, a double root:
sols = circle_parabola_intersection.match_solutions('singular_solutions');
xsols = double(sols.x); ysols = double(sols.y);
fprintf('%12s  %34s\n','x','y')
fprintf('%24.20f %24.20f\n',real([xsols ysols]).')

%%
% Only the nonsingular solutions were successfully sharpened:
s = circle_parabola_intersection.solve_summary;
istart = strfind(s,'Paths Tracked');
disp(s(istart:end))

%%
% Save for use in other examples.
save circle_parabola_intersection circle_parabola_intersection