%% Sharpening module
% Section 7.2 from _Numerically solving polynomial systems with Bertini_,
% by Daniel J. Bates, Jonathan D. Haunstein, Andrew J. Sommese and Charles
% W. Wampler (SIAM 2013).
%
% The sharpening module can be used to sharpen solutions after a homotopy
% run. Here we solve the system
%
% $$ y=x^{10}, \quad y=10^{-10}, $$
%
% which has ten nonsingular isolated solutions :
%
% $$ \left(\omega^k/10, 10^{-10}\right), $$
%
% where $\omega = \exp\left(i\pi/5\right)$ is the tenth root of unity.

poly_system = BertiniLab('function_def',{'y-x^10'; 'y-1e-10'},'variable_group',{'x','y'});
poly_system = solve(poly_system);

% Display the tally of singular and nonsingular solutions
s = poly_system.solve_summary;
istart = strfind(s,'NOTE: nonsingular');
iend = strfind(s,'Finite Multiplicity')-1;
disp(s(istart:iend))

%%
% As the above summary indicates, all 10 solutions appear to be singular.
% However, the condition numbers are all high:
sols = poly_system.match_solutions('singular_solutions');
solution_info = poly_system.read_raw_data;
condnum = [solution_info.condition_number];
xsols = double(sols.x(:)); ysols = double(sols.y(:));
fprintf('%17s %32s %22s\n','x','y','condition number')
fprintf('%15.4g + %7.4gi %15.4g + %5.4gi %15.4g\n',[real(xsols) imag(xsols) real(ysols) imag(ysols) condnum(:)].')

%%
% Now Bertini can be told to reprocess the file using the configuration
% setting |SharpenOnly| and a higher condition number threshold.
%
% Since only the configuration needs changing, we can modify the existing
% BertiniLab object.
poly_system.config = struct('SharpenOnly',1,'CondNumThreshold',1e12);


%%
% In this module, Bertini prompts the user for input. If the command
%
% |poly_system = poly_system.solve|
%
% is used, the user will be prompted for a response:
%
%   Sharpening options:
%    1. Sharpen all endpoints
%    2. Sharpen endpoints listed in a file
%    3. Manually input endpoints to sharpen
%    4. Recreate output (i.e., run the post-processor)
%    5. Change the number of sharpening digits (currently 14 digits)
%    6. Change the sharpening method (currently Newton's method)
%    7. Quit
%   Please enter your option: 
%
% If the user chooses option 4, Bertini runs the post-processor with the
% new condition number threshold.
%
% This can be automated by feeding Bertini the responses to the prompts.
% One way to do this is to give a cell array with a string for each
% response (in this case, '4') as the third argument.
poly_system.interactive_choices = {'4'};
poly_system = poly_system.solve;
% 
% fid = fopen('interactive_choices','w');
% fprintf(fid,'4\n');
% fclose(fid);
% [~,result] = system([poly_system.program_name,' input < interactive_choices']);

% Display the tally of singular and nonsingular solutions
result = poly_system.solve_summary;
istart = strfind(result,'NOTE: nonsingular');
iend = strfind(result,'Finite Multiplicity')-1;
disp(result(istart:iend))

%%
% Now Bertini recognizes that all the solutions are nonsingular.
