%% Parameter homotopy: Sextic Ab Initio
% Example 6.1 from _Numerically solving polynomial systems with Bertini_,
% by Daniel J. Bates, Jonathan D. Haunstein, Andrew J. Sommese and Charles W. Wampler (SIAM
% 2013).
%
% As a simple example of using a parameter homotopy, we solve a sextic
% polynomial in one variable. The first step is to do an _ab initio_ run to
% get the starting parameters.
%
% This example departs a little from the one in the book to illustrate how
% MATLAB handles polynomial coefficients. In the book, the parameters are

polysyms a0 a1 a2 a3 a4 a5 a6 x

% and the equation is

f = a0+x*(a1+x*(a2+x*(a3+x*(a4+x*(a5+x*a6)))));
disp(f)

%%
% This can be set up in vector form using the overloaded method POLYVAL,
% but the conventions are different in MATLAB. Indices start at 1 and the
% lowest coefficient is multiplied by the highest power:

p = polysym('a',[1 7]);
f = polyval(p,x);
disp(f)

%%
% We proceed with the _ab initio_ run.
config = struct('ParameterHomotopy',1);
poly_system = BertiniLab('function_def',f,'variable_group',x, ...
    'parameter',p,'config',config);
poly_system = solve(poly_system,'sextic.input');

%%
% Now for the parameter homotopy run:
poly_system.config = struct('ParameterHomotopy',2);

% Make nonsingular solutions the starting points
poly_system.starting_points = poly_system.read_solutions('nonsingular_solutions');

% Define final parameter values
pvals = [1.1 2.4 0.8 3.6 -0.52 -1.8 4.4];
poly_system.final_parameters = pvals;
poly_system = solve(poly_system,'sextic.input');

% Display summary of results
summary = poly_system.solve_summary;
iStart = strfind(summary,'NOTE: nonsingular');
disp(summary(iStart:end))

%%
% The solutions can be compared with the output of the MATLAB function
% ROOTS, which solves single-variable polynomials.
sols = poly_system.read_solutions('finite_solutions');
r = roots(pvals);
disp(num2str(sort([double(sols).' r])))

%%
% It would be easy to automate this process for a large number of parameter
% values, in one or more loops. The package <http://www.paramotopy.com/
% Paramotopy> is designed to do this efficiently. We hope to extend the
% BertiniLab interface to Paramotopy in a future release.
