%% The first run
% Below we show how BertiniLab can be used to set up and solve a simple
% example, the system of equations
%
% $$ x^2-1 = 0 $$
%
% $$ y^2-4 =0. $$
%
% The solution set consists of the four points:
%
% $$ (\pm 1,\pm 2). $$

%% Defining the problem
% The main input for Bertini is the |input| file. A simple example of such
% an input file is shown below:
%
%   variable_group x,y;
%   function f,g;
% 
%   f = x^2 - 1;
%   g = y^2 - 4;
%  END;
%
% Bertini is then run with the command
%
%   bertini input
% 
% A BertiniLab equivalent to the input file is shown below:
polysyms x y
f = x^2 - 1;
g = y^2 - 4;
poly_system = BertiniLab('variable_group',[x; y],'function_def',[f; g]);

%%
% The object |poly_system| looks like this:
disp(poly_system)

%%
% There are many different ways of setting up BertiniLab for the same
% problem. In this example, the variables and equations are input as cell
% arrays of strings:
poly_system = BertiniLab('variable_group',{'x';'y'},'function_def',{'x^2-1'; 'y^2-1'});

%%
% In this example, a MATLAB function is used:
polysyms x y
polyFun = @(a,b) [a^2-1; b^2-1];
poly_system = BertiniLab('variable_group',[x; y],'function_def',polyFun(x,y));

%%
% In this example, the variables and declarations are assigned separately:
poly_system = BertiniLab;
poly_system.variable_group = {'x';'y'};
poly_system.function_def = {'x^2-1'; 'y^2-1'};

%% Running Bertini
% Once the problem is defined, it is solved using the method |solve|. This
% creates the file |input| and runs Bertini.
poly_system = poly_system.solve;

%%
% Note that |poly_system| should be the output of the command (the LHS of
% the above equation) so it can store some information from the run. The
% resulting object |poly_system| is shown below:
disp(poly_system)

%%
% A lot of the properties are empty, but |function_def| and
% |variable_group| are assigned; and after |solve| was called, values are
% assigned to |order_of_variables|, |solve_summary| and
% |output_file_names|.
%
% |order_of_variables| stores the order in which the variables were
% declared in the Bertini input file:
disp(poly_system.order_of_variables)

%%
% |output_file_names| retrieves the names of the files that were created by
% Bertini during its run:
disp(poly_system.output_file_names(:))

%%
% Finally, |solve_summary| contains the summary of the run that would go to
% the screen if Bertini were called in a command window:
disp(poly_system.solve_summary)

%% Retrieving the solutions
%
% The output summary explains the outputs of Bertini. If the users wishes
% to simply retrieve the solution points, all of them are contained in
% |raw_solutions|. Since the solutions are real, nonsingular and finite,
% they should all be in |finite_solutions|, |real_finite_solutions| and
% |nonsingular_solutions|. The method |match_solutions| retrieves the data
% and assigns it to the correct variables:
sols = poly_system.match_solutions('real_finite_solutions');
disp(sols)

%%
% The solutions are:
disp([sols.x sols.y])

%%
% That's a bit messy. Since the solutions are classified as real, the
% imaginary part can be ignored. The class |polysym| has the method
% |real_imag| to do this, and we can add the variable names as titles by
% putting the |polysym| variables in the top row:
[x_re,~] = sols.x.real_imag;
[y_re,~] = sols.y.real_imag;
disp([x x_re; y y_re].')

%%
% The solutions can also be converted to other classes such as double
% precision:
disp(double([x_re y_re]))
