%% BertiniLab: User Guide
%
% The BertiniLab package is designed to convert numerical MATLAB(R)
% functions into text files that can be used as input to the numerical
% algebraic geometry package Bertini. This package can find all the roots
% of systems of polynomial equations. The package includes two classes:
% <matlab:doc('polysym') polysym> and <matlab:doc('BertiniLab') BertiniLab>. 
%
% The class |polysym| implements polynomial calculations in a symbolic
% form. It is similar to the <matlab:doc('sym') sym> class in the MATLAB
% symbolic toolbox, but tailored to the requirements of Bertini. If the
% inputs to a function are defined as |polysym| objects, the output will
% also be a |polysym| object.
%
% Given a function and arguments, definitions and configurations defined as
% |polysym| objects, BertiniLab can be used to create the input file for
% Bertini, run Bertini, and collect the outputs.
%
% This guide concentrates on how to use BertiniLab as an interface with
% Bertini. For how Bertini is used, see _Numerically solving polynomial
% systems with Bertini_, by Daniel J. Bates, Jonathan D. Haunstein, Andrew
% J. Sommese and Charles W. Wampler (SIAM 2013). The examples from the book
% are also implemented in BertiniLab: type |demo toolbox BertiniLab| to
% access them.

%% Creating a BertiniLab object
%
% An empty BertiniLab object is created by the call
poly_system = BertiniLab;
disp(BertiniLab)

%%
% Most of the properties correspond to Bertini declarations. In particular,
% <matlab:doc('BertiniLab/variable_group') variable_group> is the most
% common property for assigning variables. These can be assigned in a
% statement like
poly_system.variable_group = 'x';

%% 
% Similarly, the equation(s) to be solved are assigned to the property
% <matlab:doc('BertiniLab/function_def') function_def>:
poly_system.function_def = 'x^2-1';

%%
% Equivalently, these properties can be defined in the initial call to
% BertiniLab using name/value pairs:
poly_system = BertiniLab('variable_group','x','function_def','x^2-1');

%% Defining the variables
%
% Bertini solves a system of polynomial equations, so it needs a list of
% variables and the equations in those variables.
%
% Depending on the problem, the following properties can be used for
% defining variables:
% 
% * <matlab:doc('BertiniLab/variable_group') variable_group> (the most
% common)
% * <matlab:doc('BertiniLab/hom_variable_group') hom_variable_group> (for
% homogeneous problems)
% * <matlab:doc('BertiniLab/hom_variable_group') variable> (for
% user-defined homotopies)

%%
% As an illustration, we will solve a generalized eigenvalue problem 
%
% $$\kappa A v = \lambda B v. $$
%
% If the matrices are 2x2, the variables are $\kappa$, $\lambda$, $v_1$ and
% $v_2$. For BertiniLab, these can be defined as a cell array of strings:
vars = {'kappa','lambda','v1','v2'};

%%
% Alternatively, the variables can be defined as |polysym| objects:
polysyms kappa lambda v1 v2
vars = [kappa lambda v1 v2];

%%
% Using |polysym|, the components of |v| can be defined in vector form:
v = polysym('v',[1 2]);
vars = [kappa lambda v];

%%
% In this example, all the terms in the equations are of degree 2 (a
% constant times two variables), so the equations are homogeneous. The
% variables are therefore assigned to the property
% <matlab:doc('hom_variable_group') hom_variable_group>:
poly_system = BertiniLab('hom_variable_group',vars);

%%
% It is better to use multihomogeneous homotopy to solve this problem. This
% involves grouping variables in a way that reduces the number of paths
% that Bertini must track. We group |kappa| and |lambda| by putting them in
% a vector |[kappa lambda]|, and combine it with the vector |v| in a cell
% array:
poly_system = BertiniLab('hom_variable_group',{[kappa lambda],v});

%%
% BertiniLab also accepts <matlab:doc('vpa') variable precision> and
% <matlab:doc('sym') symbolic> objects, if the user has the Symbolic
% Toolbox.

%% Defining the equations
% Equations are defined in the property
% <matlab:doc('BertiniLab/function_def') function_def>. We will look at a
% specific example of the matrices:
%
% $$A = \left[\begin{array}{cc} 1 & 2 \\ 2 & 4 \end{array}\right]$$
%
% $$B = \left[\begin{array}{cc} 4 & -2 \\ -2 & 1 \end{array}\right].$$
%
% Written out in full, the resulting equations are
%
% $$ \kappa \left(v_1 + 2 v_2\right) - \lambda \left(4 v_1 - 2 v_2\right) = 0 $$
%
% $$ \kappa \left(2 v_1 + 4 v_2\right) = \lambda \left(-2 v_1 + v_2\right)= 0. $$
%
% There are three basic ways to define the equations for BertiniLab to solve:
% 
% # Write out the equations explicitly
% # Use a pre-defined function
% # Create a subclass

%%
% *Write out the equations explicitly*
%
% One way to define the equations is as a cell array of strings:
eqns = {'kappa*(v1+2*v2) - lambda*(4*v1-2*v2)'; 'kappa*(2*v1+4*v2) - lambda*(-2*v1+v2)'};
disp(eqns)

%%
% If the variables are |polysym| objects, the functions can be defined
% using matrix algebra:
A = [1 2; 2 4]; B = [4 -2; -2 1];
eqns = kappa*A*v.' - lambda*B*v.';

%% 
% *Using a pre-defined function*
%
% The equations can also be defined using a function call. This function
% can be pre-defined. For example, suppose we have the function
f = @(X,Y,m,l,w) (m*X-l*Y)*w.';

%%
% We can now define the variables and produce expressions for the
% equations.
polysyms kappa lambda
A = [1 2; 2 4]; B = [4 -2; -2 1];

%%
% A and B will be determined by the physical problem. Here we'll use the
% same A and B, but formulate the problem so that larger matrices could be
% substituted.
N = length(A);
v = polysym('v',[1 N]);
eqns = f(A,B,kappa,lambda,v);

%%
% The Bertini input file expects each equation to have a name. By default,
% BertiniLab generates the names. If the user wishes to provide the names,
% they can use the property <matlab:doc('BertiniLab/function_name')
% function_name>. For example, to call the equations |f1|, |f2|, etc., use
% the following:
poly_system.function_name = polysym('f',[1 N]);

%% Solving BertiniLab: Isolated points
%
% We will now combine the variable and function definitions and solve for
% B.
vars = {[kappa lambda],v};
poly_system = BertiniLab('function_def',eqns,'hom_variable_group',vars);
poly_system = solve(poly_system);

%%
% The above code creates an input file for Bertini with the following content:
type('./input')

%%
% Of course, for this example it would be just as easy to create the file
% by hand; but in creating it, BertiniLab also stores some information that
% makes it easy to import the solutions into MATLAB and analyze them.
%
% After creating the input file (and any other necessary data files), the
% <matlab:doc('solve') solve> method runs Bertini, which creates several output files.
%
% Note that you need to assign the output of the solve to |poly_system|.
% This saves information that is needed to access and interpret the output
% of Bertini. In addition, the property |solve_summary| contains the
% diagnostic output from Bertini:
disp(poly_system.solve_summary)

%% Retrieving the solutions
% The method <matlab:doc('BertiniLab/read_raw_data') read_raw_data> reads
% the diagnostic information for the solutions. One solution is displayed
% below.
diagnostics = poly_system.read_raw_data;
disp(diagnostics(1))

%%
% The method <matlab:doc('BertiniLab/read_solutions') read_solutions> can
% be used to read most other data files:
sols = poly_system.read_solutions('nonsingular_solutions');
disp(sols)

%%
% The solutions are saved as |polysym| arrays, so they retain the full
% accuracy of the Bertini solutions. The class |polysym| has the method
% <matlab:doc('BertiniLab/real_imag') real_imag> to extract the real and
% imaginary parts of the solution:
[re,im] = sols.real_imag;
disp(re)

%%
% Solutions can also be converted to <matlab:doc('polysym/double') double
% precision>, <matlab:doc('polysym/vpa') variable precision> or
% <matlab:doc('polysym/sym') symbolic> objects.
disp(re.double)

%%
disp(re.vpa)

%%
% The data can also be matched with the variables using
% <matlab:doc('BertiniLab/match_solutions') match_solutions>. This puts the
% data into a structure with a field for each variable, named after that
% variable. Vector and matrix variables are collected in arrays of
% appropriate size.
sols = poly_system.match_solutions('nonsingular_solutions');
disp(sols)
disp(sols.v)

%%
% This structure can be converted back to |polysym| arrays using
% <matlab:doc('BertiniLab/struct2mat') struct2mat>.
sols = poly_system.struct2mat(sols);
disp(sols)

%%
% This format is needed for the data in some applications where Bertini
% reads additional data files.

%% Using a subclass
%
% BertiniLab is a Matlab class, so the user can create subclasses to modify
% its behavior or apply it to particular kinds of problem. The following
% code (which can be found in the file |generalized_eigenvalue_problem.m|)
% constructs a subclass of BertiniLab that takes just two arguments, the
% matrices |A| and |B|, finds the generalized eigenvalues |mu| and
% |lambda|, and saves them as properties of the subclass. Note that the
% method |solve| is overloaded for this purpose.
%

%%
%   classdef generalized_eigenvalue_problem < BertiniLab
%      properties
%         kappa = [];
%         lambda = [];
%      end
%      methods
%         function obj = generalized_eigenvalue_problem(A,B)
%             m = polysym('kappa');
%             l = polysym('lambda');
%             N = length(A);
%             v = polysym('v',[1 N]);
%             obj = obj@BertiniLab('function_def',(m*A-l*B)*v.','hom_variable_group',{[m l],v});
%             obj.kappa = m;
%             obj.lambda = l;
%         end
%         function obj = solve(obj)
%             obj = solve@BertiniLab(obj);
%             kappa = obj.kappa; lambda = obj.lambda;
%             sstruct = obj.match_solutions('nonsingular_solutions','kappa','lambda');
%             obj.kappa = sstruct.kappa;
%             obj.lambda = sstruct.lambda;
%         end
%      end
%   end
%
% Here is the code to run it:
A = [1 2; 2 4]; B = [4 -2; -2 1];

poly_system = generalized_eigenvalue_problem(A,B);
poly_system = poly_system.solve;
disp([poly_system.kappa poly_system.lambda])

%% Solving BertiniLab: finite-dimensional solutions
%
% The above methods are for finding isolated points that are solutions of a
% polynomial system, also known as zero-dimensional solutions. However, a
% system may also have solutions that include curves, surfaces and other
% positive-dimensional objects. Bertini has extensive tools for exploring
% such solutions. In particular, it finds the _irreducible decomposition_
% of an algebraic set by removing singular points, identifying the
% connected components, and putting the singular points back in. For
% example, the system of polynomial equations
%
% $$ x*z + y = 0 $$
%
% $$ y*z + x = 0 $$
%
% has an irreducible solution set consisting of three lines: one with $x =
% y = 0$ and $z$ free, one with $x = y$ (nonzero) and $z = -1$; and one
% with $x = -y$ (nonzero) and $z = 1$.
%
% With suitable configurations, the command |solve| can be used to find
% irreducible decompositions, but BertiniLab makes it easier with the
% method <matlab:doc('BertiniLab/irreducible_decomposition') irreducible_decomposition>:
polysyms x y z
poly_system = BertiniLab('function_def',[x*z+y; y*z+x], ...
    'variable_group',[x y z]);
poly_system = poly_system.irreducible_decomposition;

%%
% A lot of useful information about the decomposition is contained in
% |solve_summary|:
results = poly_system.solve_summary;
istart = strfind(results,'************** Decomposition by Degree');
disp(results(istart:end))

%%
% For each component of the solution set, Bertini returns a _witness point_,
% a generic point from that set. Information about each point (including dimension
% and degree) is returned in the property <matlab:doc('BertiniLab/components') components>.
disp(poly_system.components)

%%
% The witness points themselves are returned in
% <matlab:doc('BertiniLab/witness_points') witness_points> as a structure with one
% element per component. Each structure has fields corresponding to the
% names of the variables and any parameters that are allowed to vary. It
% has one witness point for each degree.
disp(poly_system.witness_points)

%%
% The method <matlab:doc('BertiniLab/membership') membership> can be used to test
% proposed solutions for membership. It returns a logical array with a
% column for each point and a row for each component. An element is true if
% the point is a member of that component. The order of components varies,
% but if we input a point from each of the three solutions, we will get a
% permutation of the identity matrix.
membership_info = poly_system.membership([0 0 pi; 2 2 -1; 2 -2 1].');
disp(membership_info)

%%
% A component can also be sampled using the method <matlab:doc('BertiniLab/sample')
% sample>. The dimension and component number must be provided in a
% structure, information that can be obtained from |components|. An
% alternative is simply to input one of the elements of |components|:
sample_points = poly_system.sample(poly_system.components(1),100); % sample 100 points
disp(sample_points)

%% Shortcuts and utilities
%
% Some simple shortcuts are available for using BertiniLab.

%%
% The function <matlab:doc('polysolve') polysolve> has a syntax similar to
% that of the Symbolic Math ToolBox function <matlab:doc('solve') solve>.
% For example, to solve the system of equations
%
% $$ x^2-1 = 0 $$
%
% $$ y^2-4 =0, $$
%
% for which the solution set consists of the four points
%
% $$ (\pm 1,\pm 2), $$
%
% the following command can be used:
sols = polysolve({'x^2-1'; 'y^2-4'});
disp(sols)

%%
% The solutions are
disp([sols.x sols.y])

%%
% The function <matlab:doc('findRealRoots') findRealRoots> has a similar
% syntax to |polysolve| but returns the real roots. For example,
sols = findRealRoots({'x^2-1'; 'y^2-4'});
disp([sols.x sols.y])

%%
% The function <matlab:doc('BertiniLab/BertiniClean') BertiniClean> removes all the
% Bertini-generated files from a folder.

%% Configuring BertiniLab

%%
% *Choosing an I/O folder:*
%
% BertiniLab creates input files for Bertini and reads output files. By
% default, all the files go in the current folder (which can be identified
% using <matlab:doc('pwd') pwd>). However, the user can change this using
% the property <matlab:doc('Bertini/Bertini_io_folder') Bertini_io_folder>.
% For example, the following commands provide the full path name for a
% folder into which all the files will go.
poly_system = BertiniLab;
poly_system.Bertini_io_folder = '/Users/my_name/matlab/Bertini_IO';

%%
% A relative path name can be provided instead. The following will put all
% the files in a subfolder |Bertini_IO| of the current folder (creating
% |Bertini_IO| if necessary).
poly_system.Bertini_io_folder = './Bertini_IO';

%%
% The above changes only last as long as the BertiniLab object exists. If
% the user wants to use the same IO folder for all BertiniLab runs, they
% can edit the line |Bertini_io_folder = ''|; in the file |BertiniLab.m|.
%

%%
% *Monitoring a Bertini run:* 
%
% Bertini reports stages in the progress of a run (for example, how many
% curves have been traced). By default, this information is stored in the
% variable <matlab:doc('BertiniLab/solve_summary') solve_summary>. However,
% if the sets the property <matlab:doc('BertiniLab/view_progress')
% view_progress> to |true|, this information will be echoed to the command
% window, so the user can monitor the progress of the calculation:
poly_system = BertiniLab;
poly_system.view_progress = true;

%% Learning more about BertiniLab
% 
% Some users may not be familiar with object-oriented programming in
% MATLAB. BertiniLab is a _class_, and an instance of BertiniLab is an
% _object_. The object-oriented analog of functions is a _method_. To see
% an overview of the methods in BertiniLab, type
methods BertiniLab

%%
% Variables in a class are called _properties_. To see the properties in
% BertiniLab, type
properties BertiniLab

%%
% Information on any method or property can be viewed in the command window
% using the |help| function, with |BertiniLab/| preceding the name:
help BertiniLab/config

%%
% It can also be viewed in the Help browser using |doc BertiniLab/config|.
