function [re,im] = real_imag(p)
%REAL_IMAG Separate real and imaginary parts of complex numbers.
%
% [re,im] = real_imag(p) takes a POLYSYM array and returns a POLYSYM array
% for each of the real and imaginary parts.
%
% Numbers should be in the form A + B*I (spaces not important), where A and
% B are numbers in decimal or scientific notation (e.g., 3.14159 or
% 1.4e-10). If either the real or imaginary part is missing, it is
% represented by a zero.
%
% See also: double
% 
% Example:
%   x = polysym(exp(1i*randn(2)));
%   [re,im] = real_imag(x);

validateattributes(p,{'polysym'},{},'write_real_imag','p',1)

% Default value is 0.
re = polysym(zeros(size(p))); im = re;

S = cellstr(p);
% For each string, find the imaginary part (if any) and remove it. The
% remainder should be the real part (with leading and trailing blanks
% allowed).
for ii=1:numel(S)
    Sii = S{ii};
    
    % Imaginary part
    [matchstrings,Sii] = polysym.find_imag_numbers(Sii);
    assert(numel(matchstrings)<=1,'polysym:write_number:tooMany', ...
        'Expression for number has more than one imaginary term.')
    if ~isempty(matchstrings)
        im(ii) = polysym(matchstrings);
    end
    
    % Real part
    [matchstrings,Sii] = polysym.find_real_numbers(Sii);
    assert(numel(matchstrings)<=1,'polysym:write_number:tooMany', ...
        'Expression for number has more than one real term.')
    assert(isempty(Sii),'polysym:write_number:notNumber', ...
        'Attempted to read number but not a valid expression.')
    if ~isempty(matchstrings)
        re(ii) = polysym(matchstrings);
    end
end
