function y = diff(x,n,dim)
%DIFF Differences between adjacent terms
%
% y = diff(x) calculates differences between adjacent elements of X.
%
% If X is a vector, diff(x) returns a vector:
%   (x(2)-x(1), x(3)-x(2), ..., x(n)-x(n-1))
%
% If X is a matrix, diff(x) returns the analogous differences between rows.
%
% y = diff(x,n) applies diff recursively:
%   diff(x,n) = diff(diff(x,n-1));
%
% y = diff(x,n,dim) calculates the difference along dimension DIM.
%
% If n equals or exceeds the length of the relevant dimension, it returns an
% empty array.

if nargin < 2
    n = 1;
end
validateattributes(n,{'numeric'},{'positive','integer','scalar'}, ...
    'polysym:diff','n',2)

if nargin < 3
    dim = 1;
    if isrow(x)
        dim = 2;
    end
end
validateattributes(dim,{'numeric'},{'integer','scalar','>',0,'<',3}, ...
    'polysym:diff','dim',3)

if n >= size(x,dim)
    y = [];
    return
elseif n > 1
    x = diff(x,n-1,dim);
end

switch dim
    case 1
        y = x(2:end,:)-x(1:end-1,:);
    case 2
        y = x(:,2:end)-x(:,1:end-1);
end

