function obj = diag(obj,varargin)
%  DIAG Diagonal matrices and diagonals of a matrix.
%     obj = diag(obj,k) when OBJ is a POLYSYM vector with N components
%     returns a square POLYSYM array of order N+ABS(K) with the elements of
%     OBJ on the K-th diagonal. K = 0 is the main diagonal, K > 0 is above
%     the main diagonal and K < 0 is below the main diagonal.
%  
%     obj = diag(obj) is the same as diag(obj,0) and puts OBJ on the main
%     diagonal.
%  
%     obj = diag(obj,k) when OBJ is a POLYSYM matrix returns a column
%     vector formed from the elements of the K-th diagonal of OBJ.
%  
%     obj = diag(obj) is the main diagonal of OBJ. DIAG(DIAG(OBJ)) is a
%     diagonal matrix.
%  
%     Example
%        m = 5;
%        diag(polysym(-m:m)) + diag(polysym(ones(2*m,1)),1) + diag(polysym(ones(2*m,1)),-1)
%     produces a tridiagonal matrix of order 2*m+1.

n = length(obj);
k=0;
if nargin > 1
    k = varargin{1};
end
nk = n + abs(k);

% Calculate indices for diagonal
idx = [1:n; abs(k)+(1:n)].';
if k < 0
    idx = idx(:,[2 1]);
end

if isvector(obj)
    
    out = polysym(zeros(nk));
    
    % Do positive k and then flip if necessary
    for ii=1:n
        out(idx(ii,1),idx(ii,2)) = obj(ii);
    end

elseif ismatrix(obj)
    out = polysym(zeros(n,1));
    
    for ii=1:n
        out(ii) = obj(idx(ii,1),idx(ii,2));
    end

end
   
obj = out;    