function s = cumprod(obj,dim)
%CUMPROD Cumulative product
%
% s = cumprod(obj) returns a cumulative product that depends on the shape of OBJ.
%   If OBJ is a vector, S is a vector.
%   If OBJ is a matrix, S is a matrix of the same size with the cumulative
%   products for each column of OBJ.
%
% s = cumprod(obj,dim) returns the product of elements along dimension DIM.
%
% Example:
%  s = polysym({'a','b'; 'c','d'});
%  cumprod(s)
%
% See also cumsum, prod.

if nargin < 2
    dim = 1;
    if isrow(obj)
        dim = 2;
    end
end
validateattributes(dim,{'numeric'},{'integer','scalar','>',0,'<',3}, ...
    'polysym:diff','dim',2)

switch dim
    case 1
        s = obj;
        for k=2:size(obj,1)
            s(k,:) = s(k-1,:).*obj(k,:);
        end
    case 2
        s = obj;
        for k=2:size(obj,2)
            s(:,k) = s(:,k-1).*obj(:,k);
        end
end
