function M = struct2mat(poly_sys,sstruct)
%STRUCT2MAT Turn solution structure into array for Bertini data file
%
% M = poly_sys.struct2mat(sstruct) inputs SSTRUCT, a collection of potential
% solutions for the polynomial system. Returns M, a POLYSYM matrix with a
% column for each solution and variables in the order they are declared in
% the Bertini input file.
%
% SSTRUCT may be a structure with a field name for each variable; a
% numeric/polysym/sym array; or something that may be converted into a
% polysym array.
%
% See also: mat2struct

variable_order = poly_sys.order_of_variables;
assert(~isempty(variable_order),'BertiniLab:struct2mat:noVariables', ...
    'Some variables must be declared before assigning values to starting_points or member_points.')

if ~isstruct(sstruct)
    
    % Allow a row vector for a single point
    if isrow(sstruct) && length(variable_order) > 1
        sstruct = sstruct.';
    end
    
    assert(size(sstruct,1)==length(variable_order), ...
        'BertiniLab:struct2mat:wrongLength', ...
        ['Number of components does not match number of variables. Variables are ', ...
        format_array(variable_order(:).')])
    M = polysym(sstruct);
    return
end

% Determine names of variable arrays
vars = variable_order.put_in_arrays;
vnames = cellfun(@common_name,vars,'UniformOutput',false);

% Compare with field names in the solutions structure
fnames = unique(fieldnames(sstruct)); % 'unique' is just for sorting
assert(isequal(vnames,fnames),'BertiniLab:struct2mat:mismatched', ...
    ['The variable names in the solution structure do not agree with those ', ...
    'in the BertiniLab object.'])

% Determine number of variables and number of points
num_vars = numel(variable_order);
v = vars{1};
var_name = v.char_base;
value = sstruct.(var_name{1});
num_points = numel(value)/numel(v);

M = polysym(zeros(num_vars,num_points));
for ii=1:numel(vars)
    v = vars{ii};
    
    % Extract values for variable
    var_name = v.common_name;
    value = sstruct.(var_name);
    
    % Find locations of components in the variable order
    [~,~,ix] = intersect(v,variable_order);
    M(ix,:) = reshape(value,[],num_points);
end
M = polysym(M);