function poly_sys = sharpen(poly_sys,opts)
%SHARPEN Sharpen solutions of a homotopy using the sharpening module
%
% poly_sys = poly_sys.sharpen starts an interactive dialog with Bertini's
% sharpening module.
%
% poly_sys = poly_sys.sharpen(opts) inputs a structure OPTS with one or
% more of the fields endpoint_selection, and sharpening_method. Runs
% Bertini and returns the sharpened solutions in POLY_SYS.
%
% endpoint_selection can be a vector of integers (referring to path
% numbers) or the name of a file. The default is to sharpen all points. If
% a file name is given, there should be one integer per line in that file.
%
% sharpening_method can be 'Newton' (the default) or 'Endgame' (useful for
% sharpening singular endpoints). An empty string is taken as the default.
%
% The user may also wish to modify the configuration option SharpenDigits.
%
% Example:
%       poly_system.config.SharpenDigits = 20;
%       opts = struct('sharpening_method','endgame');
%       poly_system = poly_system.solve;

poly_sys.config.SharpenOnly = 1;
interactive_selections = {'1'}; % Default: sharpen all the points.

if nargin > 1
    
    if isfield(opts,'endpoint_selection')
        es = opts.endpoint_selection;
        validateattributes(es,{'numeric','char'},{},'sharpen','opts.endpoint_selection')
        if ischar(es) % read from file
            assert(exist(poly_sys.fullname(es),'file')==2, ...
                'BertiniLab:sharpen:fileNotFound', ...
                sprintf('The file %s was not found.',es))
            interactive_selections = {'2', es};
        elseif ~isempty(es)
            interactive_selections = num2cell([3; es(:)])';
        end
    end
    
    if isfield(opts,'sharpening_method')
        switch lower(opts.sharpening_method)
            case {'','newton'}
                selection = {'6','1'};
            case 'endgame'
                cf = poly_sys.config;
                assert(~isfield(cf,'UseRegeneration') || cf.UseRegeneration == 0, ...
                    'BertiniLab:sharpen:invalidConfiguration', ...
                    'The endgame method cannot be used after regeneration is used.')
                assert(~isfield(cf,'UseDiagonal') || cf.UseDiagonal == 0, ...
                    'BertiniLab:sharpen:invalidConfiguration', ...
                     'The endgame method cannot be used after a diagonal equation-by-equation approach.')
                assert(~isfield(cf,'SharpenDigits') || cf.SharpenDigits <= 300, ...
                    'BertiniLab:sharpen:invalidConfiguration', ...
                    'The endgame method cannot be used with more than 300 sharpening digits.')
                selection = {'6','2'};
            otherwise
                error('BertiniLab:sharpen:invalidInput', ...
                    'The option sharpening_method must be ''Newton'', ''Endgame'' or an empty string.')
        end
        interactive_selections = [selection interactive_selections];
    end
end

poly_sys.interactive_choices = interactive_selections;   
poly_sys = poly_sys.solve;