function sample_points = sample(poly_sys,component,npoints,fname)
%SAMPLE Sample an algebraic component
%
% sample_points = poly_sys.sample(component,npoints) inputs COMPONENT, a
% structure with the fields DIM and COMPONENT_NUMBER (both integers); and
% NPOINTS, the number of points to be sampled (default: 1). Returns
% SAMPLE_POINTS, a structure in which each field is named after a variable
% and its value is an array of solutions for that variable. Vectors and
% matrices are placed in arrays of appropriate dimensions, with one column
% per vector and one page per matrix.
%
% sample_points = poly_sys.sample(component,npoints,fname) inputs the
% string FNAME and saves the sample from Bertini in a file of that name.
% Otherwise, the default name 'component_sample' is used.
%
% SAMPLE does not store the diagnostic output from Bertini. To see this
% output displayed in the command window, set the property VIEW_PROGRESS
% to TRUE.
%
% SAMPLE does not support the interactive version of this call to Bertini.
% If the user wishes to do this, they can use the commands:
%
%   poly_system.config = struct('TrackType',2);
%   poly_system = poly_system.solve:
%  
% If the user gives a file name FNAME for the sample points in the
% interactive dialog, they can retrieve them using
%
%   sample_points = poly_system.match_solutions(fname);
%
% See also: irreducible_decomposition, membership

assert(nargin > 1,'BertiniLab:sample:minrhs', ...
    ['Not enough arguments. The user must at least provide ', ...
    'the component information.'])
validateattributes(component,{'struct'},{'nonempty'},'sample','component',2)

% Check that component exists
acceptable_dims = unique([poly_sys.components.dim]);
assert(ismember(component.dim,acceptable_dims),'BertiniLab:sample:invalidNumber', ...
    [sprintf('A component of dimension %d does not exist. Choose one of: ', ...
    component.dim),num2str(acceptable_dims)])
idx = [poly_sys.components.dim]==component.dim;
acceptable_component_numbers = [poly_sys.components(idx).component_number];
assert(ismember(component.component_number(1),acceptable_component_numbers), ...
    'BertiniLab:sample:invalidNumber', ...
    [sprintf('Component number %d for dimension %d does not exist. Choose one of: ', ...
    component.component_number,component.dim),num2str(acceptable_component_numbers)])

% Check that number of points is valid
if nargin > 2
    validateattributes(npoints,{'numeric'},{'scalar','integer','nonnegative'},'sample','npoints',3);
else
    npoints = 1;
end

% Determine user responses (they are written to an input file for Bertini).
choices = struct('dim',num2str(component.dim), ...
    'component_number',num2str(component.component_number), ...
    'npoints',num2str(npoints),'write_to_file','0','fname','component_sample');
if nargin > 3
    choices.fname = fname;
end

poly_sys.interactive_choices = struct2cell(choices)';

poly_sys.config = struct('TrackType',2);
poly_sys = poly_sys.solve;
sample_points = poly_sys.match_solutions(choices.fname);

