function components = read_witness_data(poly_sys)
%READ_WITNESS_DATA Read information on witness points
%
% components = poly_sys.read_witness_data reads the file witness_data,
% containing witness points for a positive-dimensional run and diagnostic
% information, and returns the data in the structure array COMPONENTS.
%
% Each element of COMPONENTS contains all the witness points for an
% algebraic component. The fields are:
%   codim -                         integer (counts up from 0)
%   dim -                           integer (derived from codim)
%   degree -                        integer (polynomial degree of component)
%   endpoint -                      polysym array with one column per solution
%   last_approx -                   polysym array with one column per approximation
%   endpoint_precision -            precision used for the endpoint
%   last_approx_precision -         precision used for last approximation
%   condition_number -              double precision
%   corank -                        integer
%   smallest_singular -             double precision
%   largest_singular -              double precision
%   type -                          integer
%   multiplicity -                  integer
%   component_number -              integer
%   deflations_needed -             integer
%
% The format of witness_data is defined in Appendix D of "Numerically
% solving polynomial systems with Bertini", by Daniel J. Bates, Jonathan D.
% Haunstein, Andrew J. Sommese and Charles W. Wampler (SIAM 2013).
%
% The solutions are in homogeneous coordinates. To convert to affine
% coordinates, use DEHOMOGENIZE.
%
% See also: solve, read_solutions, match_solutions

% Finite solutions
fname = poly_sys.fullname('witness_data');
[fid,msg] = fopen(fname,'r');
if fid < 3
    error('BertiniLab:read_witness_data:openError', msg)
end

nVariables = fscanf(fid,'%d\n',1);
nCodims = fscanf(fid,'%d\n',1);

% Fill in structure backwards so it is pre-allocated.
for jj = nCodims:-1:1
    solution_info(jj).codim = fscanf(fid,'%d\n',1);
    solution_info(jj).dim = nVariables - solution_info(jj).codim - 1;
    nPoints = fscanf(fid,'%d\n',1);
    
    % Read all the witness points for a given codim
    for kk = 1:nPoints
        solution_info(jj).endpoint_precision(kk) = fscanf(fid,'%d\n',1);
        solution_info(jj).endpoint(:,kk)  = BertiniLab.read_data_point(fid,nVariables);
        solution_info(jj).last_approx_precision(kk) = fscanf(fid,'%d\n',1);
        solution_info(jj).last_approx(:,kk)  = BertiniLab.read_data_point(fid,nVariables);
        solution_info(jj).condition_number(kk) = fscanf(fid,'%f\n',1);
        solution_info(jj).corank(kk) = fscanf(fid,'%d\n',1);
        solution_info(jj).smallest_singular(kk) = fscanf(fid,'%f\n',1);
        solution_info(jj).largest_singular(kk) = fscanf(fid,'%f\n',1);
        solution_info(jj).type(kk) = fscanf(fid,'%d\n',1);
        solution_info(jj).multiplicity(kk) = fscanf(fid,'%d\n',1);
        solution_info(jj).component_number(kk) = fscanf(fid,'%d\n',1);
        solution_info(jj).deflations_needed(kk) = fscanf(fid,'%d\n',1);
    end
end

% Split codims into components
iComponents = arrayfun(@(x) max(x.component_number),solution_info) + 1;
components = solution_info(1);
components.degree = 1;
components = repmat(components,1,sum(iComponents));

runningTotal = cumsum(iComponents);
runningTotal = [0 runningTotal(1:end-1)];
names = fieldnames(solution_info);
for jj=1:nCodims
    for kk = 1:iComponents(jj);
        idx = solution_info(jj).component_number==kk-1;
        c.codim = solution_info(jj).codim;
        c.dim = solution_info(jj).dim;
        for ii = 3:length(names)
            c.(names{ii}) = solution_info(jj).(names{ii})(:,idx);
        end
        c.degree = sum(idx);
        components(runningTotal(jj)+kk) = c;
    end
end

fclose(fid);
