function data = read_solutions(poly_sys,fname,nComponents)
%READ_SOLUTIONS Read raw_solutions or other files with same format
%
% data = poly_sys.read_solutions(fname) inputs the string FNAME and returns
% DATA, a POLYSYM array containing the data from the file by that name.
% DATA has one column for each solution.
%
% data = poly_sys.read_solutions(fname,nComponents) inputs NCOMPONENTS, the
% number of components of the data vector for each point.
%
% The data file is assumed to have the number of solutions in the first
% line (an integer), followed by a block for each solution. If the file
% name is 'raw_solutions', the blocks have the format:
%   (blank line)
%   solutionNumber (integer)
%   real(first_coordinate) imag(first_coordinate)
%         ...               ...
%   real(last_coordinate) imag(last_coordinate)
%
% For any other file name, the solutionNumber line is omitted in each
% block.
%
% Example:
%   poly_system = BertiniLab('variable_group','x','function_def','x^2-1');
%   poly_system = solve(poly_system);
%   data = poly_system.read_solutions('raw_solutions');
%
% See also: solve, read_raw_data, match_solutions
%
% Source: Appendix D of "Numerically solving polynomial systems with
% Bertini", by Daniel J. Bates, Jonathan D. Haunstein, Andrew J. Sommese
% and Charles W. Wampler (SIAM 2013)

% Determine number of variables
if nargin < 3
    nComponents = numel(poly_sys.order_of_variables);
end

% Finite solutions
fname = poly_sys.fullname(fname);
[fid,msg] = fopen(fname,'r');
assert(fid >= 3,'BertiniLab:read_solutions:openError', sprintf('Error opening %s:',fname,msg))

nPoints = fscanf(fid,'%d\n\n',1);
if isempty(nPoints)
    nPoints=0;
end


% Determine whether file has path numbers (raw_solutions)
hasPathNumber = ~isempty(strfind(fname,'raw_solutions'));

data = polysym(zeros(nComponents,nPoints));
for ii=1:nPoints
    if hasPathNumber
        solutionNumber = fscanf(fid,'%d\n',1);
        assert(solutionNumber==ii-1,'BertiniLab:read_solutions:wrongNumber', ...
            sprintf('Expected solution number %d, received %d.\n %s', ...
            ii-1,solutionNumber,poly_sys.solve_summary))
    end
    data(:,ii) = BertiniLab.read_data_point(fid,nComponents);
    fgetl(fid);
end
fclose(fid);
