function out = format_struct(in)
%FORMAT_STRUCT Convert structure to appropriate string form
%
% out = format_struct(in) inputs a structure IN and returns a cell array of
% strings in (name,value) pairs. Each field of the structure may be a
% numeric value or a cell array of POLYSYM objects.
%
% Example:
%   s = struct('variable_group',polysym({'x','y'}));
%   BertiniLab.format_struct(s)
%
% See also makeInputFile

validateattributes(in,{'struct'},{},'BertiniLab.format_struct','in')

names = fieldnames(in);
vals = struct2cell(in);

count = cellfun(@numel,vals,'UniformOutput',0);
count = [count{:}];

out = {};

for ii=1:numel(count)
    val = vals{ii};
    if isnumeric(val)
        val = {num2str(val)};
    elseif isa(val,'polysym')
        val = {format_array(val(:).',', ')};
    elseif iscellpolysym(val)
        val = cellfun(@(x) format_array(x,', '),val(:).','UniformOutput',false);
    else
        error('BertiniLab:format_struct:invalidType', ...
            'Structure has field of invalid type.')
    end
    out = [out; repmat(names(ii),numel(val),1) val(:)]; %#ok<AGROW>
end

function tf = iscellpolysym(s)
%ISCELPOLYSYM True for cell array of POLYSYM objects.
%   iscellstr(S) returns 1 if S is a cell array of POLYSYM objects and
%   0 otherwise.
%
% Example:
%   s = {polysym('x'),polysym('y')};
%   tf = iscellstr(s);

if isa(s,'cell'),
  res = cellfun('isclass',s,'polysym');
  tf = all(res(:));
else
  tf = false;
end
