/****************************************************************************

SIMPLE EXAMPLE PROGRAM TO USE THE ROUTINE ReLaTIve
FOR THE NUMERICAL INVERSION OF A LAPLACE TRANSFORM

THE LAPLACE TRANSFORM FUNCTION IS fzTest, DEFINED IN THIS FILE

*******************************************************************************/

/************************************************************************\
*   Authors:
*       Luisa D'Amore - University of Naples, Federico II
*		Rosanna Campagna - University of Naples, Federico II
*		Valeria Mele - University of Naples, Federico II
*		Almerico Murli - SPACI - NAPLES
* 		Mariarosaria Rizzardi - University of Naples, Parthenope
\************************************************************************/

#include "../SRC/ReLaTIve.h"

/* F(z)=1/(z^4-a^4)  with a=3/5   the convergence abscissa is a=3/5=0.6*/
double fzTest(double z){
	return 1./(pow(z,4)-pow(3./5.,4));
}

int main(){
    /*ReLaTIve INPUT*/
	double		*x=NULL;			/*Inverse Function evaluation point(s)*/
	double		(*fz)(double);   		/*Function F Pointer*/
	int          	sinf=0; 			/*optional: there's or not a singularity at infinity for F*/
	int		sflag=1;			/*optional: user will provide abscissa of convergence*/

    /*ReLaTIve INPUT-OUTPUT*/
	double		sigma0=0.6;   			/*optional: abscissa of convergence of F */
	double 		tol=0;				/*optional: tolerance on accuracy */
	int		nmax=0; 			/*optional: maximum number of series coefficients*/

    /*ReLaTIve OUTPUT*/
	int		nopt;
	int		ncalc;
	double		*absesterr=NULL;		/*absolute error estimate*/
	double		*relesterr=NULL;		/*relative error estimate	*/
	double 		ILf;				/*Inverse Function f computed */
	int		*flag=NULL;			/*diagnostics on the result */
	int		ierr=0;				/*diagnostics on the software work */

    /*AUXILIARY VARIABLES*/
	int		dim=0;
	int		i;
	double		a,b,step;
	int		start=0;

	/* --------------------------------- x values*/
	a=1.;
	b=15.;
	step=0.5;
	dim=(int)(((b-a)/step)+1);
	x = (double*)calloc(dim,sizeof(double));
	if ( x == NULL )
    {   fprintf(stderr, "\nERROR: DYNAMIC ALLOCATION FAILED.\n");
        exit(1);
    }
    for (i=0; i<dim; i++)
        x[i] = a + i*step;
	/* ------------------------------------------*/

	/* ----------------- memory space allocations*/
	flag=(int*)calloc(dim,sizeof(int));
	if ( flag == NULL )
    {   fprintf(stderr, "\nERROR: DYNAMIC ALLOCATION FAILED.\n");
        exit(1);
    }
	relesterr=(double*)calloc(dim,sizeof(double));
	if ( relesterr == NULL )
    {   fprintf(stderr, "\nERROR: DYNAMIC ALLOCATION FAILED.\n");
        exit(1);
    }
	absesterr=(double*)calloc(dim,sizeof(double));
	if ( absesterr == NULL )
    {   fprintf(stderr, "\nERROR: DYNAMIC ALLOCATION FAILED.\n");
        exit(1);
    }
	/* ------------------------------------------*/

	fz = fzTest;

	for(i=0; i<dim; ++i){
		/************************************** CALLING ReLaTIve *********************************************/
		ierr=ReLaTIve(x[i],fz,sinf,&sigma0,sflag,&tol,&nmax,&nopt,&ncalc,&absesterr[i],&relesterr[i],&flag[i],&ILf);
		/******************************************************************************************************/

		switch(ierr){
		case 1:
			printf("\nx=%f - ReLaTIve termined: x<0.0!\n It must be x>=0\n\n",x[i]);
			break;
		default:
			if (!start){
				start=1;
				printf("Used Tolerance on accuracy: %e;\n",tol);
				printf("Used abscissa of convergence on F: %e;\n",sigma0);
				printf("Used maximum number of Laguerre series coefficients: %d;\n",nmax);
				printf("\n                             TABLE\n");
				printf("----------------------------------------------------------------------------\n");
				printf("|     x    |     f_comp     | estabserr | estrelerr |  Nopt |  Ncal |FLAG|\n");
				printf("----------------------------------------------------------------------------\n");
			}
			printf("| %4.2e | %14.8e | %9.3e | %9.3e | %5d | %5d | %2d |\n", x[i], ILf, absesterr[i], relesterr[i], nopt, ncalc, flag[i]);
			break;
		}/*endswitch*/
		/* UPDATE OF i (and x)*/
	}/*endfor on x*/

	free(flag);
	free(relesterr);
	free(absesterr);
	return 0;
}
/*END OF MAIN*/



