/************************************************************************\
*   Authors:                                                            
*       Luisa D'Amore - University of Naples, Federico II	            
*		Rosanna Campagna - University of Naples, Federico II	        
*		Valeria Mele - University of Naples, Federico II	            
*		Almerico Murli - SPACI - NAPLES                   	            
* 		Mariarosaria Rizzardi - University of Naples, Parthenope        
\************************************************************************/

#include "ReLaTIve.h"

int ck_calc(int n,double sigma,double sigma0,double b,double (*fz)(double),double *ck,double *err, int sinf){

/* =============================================================================
    1.  PURPOSE
   =============================================================================

   CK_CALC COMPUTES THE COEFFICIENTS OF THE LAGUERRE SERIES EXPANSION
   AND PROVIDES THE ESTIMATE OF THE COMPONENT-WISE ERROR ON THEM. THE COMPUTATION
   OF THE COEFFICIENTS IS BASED ON THE BJORCK-PEREIRA ALGORITHM.

   =============================================================================
    2. INPUT PARAMETERS
   =============================================================================

   n       -   (INT)      NUMBER OF COEFFICIENTS
   sigma   -   (DOUBLE)   PARAMETER GREATER THAN THE ABSCISSA OF CONVERGENCE
   sigma0  -   (DOUBLE)   THE ABSCISSA OF CONVERGENCE
   b       -   (DOUBLE)   PARAMETER
   fz      -   (FUNCTION POINTER) LAPLACE TRANSFORM TO INVERT. fz RETURNS THE VALUE
                          OF THE LAPLACE TRANSFORM FUNCTION TO BE INVERTED. IT IS
						  SUPPLIED BY THE USER.

   =============================================================================
    3. OUTPUT PARAMETERS
   =============================================================================

   ck   - (DOUBLE ARRAY)    COEFFICIENTS OF THE LAPGUERRE POLYNOMIAL SERIES

   err  - (DOUBLE ARRAY)    COMPONENT-WISE ERROR ESTIMATE ON EACH COEFFICIENT


   =============================================================================
     4. PROCEDURE NEEDED

   void diffdiv(int, double *, double *, double *);

   =============================================================================

    5. AUTHORS

    Luisa D'Amore - University of Naples, Federico II
    Rosanna Campagna - University of Naples, Federico II
    Valeria Mele - University of Naples, Federico II
    Almerico Murli - SPACI - NAPLES
    Mariarosaria Rizzardi - University of Naples Parthenope

    ===========================================================================

    6.  REFERENCES
   =============================================================================

    N. Higham,  Num. An.,1988.

    A. Bjork V. Pereira, Math. Comp., 24, 1970.

    A. Murli, S.Cuomo, L. D'Amore, M. Rizzardi - JCAM, Vol 198, N.1, 2007


    ***************************************************************************/


   /* ...Local Variables specification... */


   int		i,j,k;
   double	pi2,arg,x,y,*z;
   double	*y1,temp,temp1;
   double	*MM, *DD;
   double 	shift=-1;

   /*-------------------------------------------------------------------------*/
   /*                 MEMORY ALLOCATION                                       */
   /*-------------------------------------------------------------------------*/

    z = (double *)calloc(n, sizeof(double));
	if ( z == NULL )
    {   fprintf(stderr, "\nERROR: DYNAMIC ALLOCATION FAILED.\n");
        exit(1);
    }
    y1 = (double *)calloc(n, sizeof(double));
	if ( y1 == NULL )
    {   fprintf(stderr, "\nERROR: DYNAMIC ALLOCATION FAILED.\n");
        exit(1);
    }
    MM = (double *)calloc(n, sizeof(double));
	if ( MM == NULL )
    {   fprintf(stderr, "\nERROR: DYNAMIC ALLOCATION FAILED.\n");
        exit(1);
    }
    DD  = (double *)calloc(n, sizeof(double));
	if ( DD == NULL )
    {   fprintf(stderr, "\nERROR: DYNAMIC ALLOCATION FAILED.\n");
        exit(1);
    }

   /*-------------------------------------------------------------------------*/
   /*                  COMPUTATION OF THE CHEBYSCHEV'S ZEROES                 */
   /*-------------------------------------------------------------------------*/
   pi2= 2.* atan(1);
   for(k=0; k<n ;k++){
      arg= (double)(2*k+1)/n ;
      z[k]=cos(arg*pi2);
      x=(2*b)/(1-z[k]);
      y=x+sigma-b;
	  if (sinf) y1[k]=x*fz(y+sigma0+shift);
	  else y1[k]=x*fz(y+sigma0);
   }

   /*-------------------------------------------------------------------------------*/
   /*                   BJORCK - PEREIRA ALGORITHM FOR THE COMPUTATION OF THE       */
   /*                                 COEFFICIENTS C_K                              */
   /*-------------------------------------------------------------------------------*/

   diffdiv(n,y1,z,DD);
   for(i=0;i<n;i++){
      MM[i]=DD[i];
      ck[i]=y1[i];
   }
   k=n-1;
   for(k=n-1;k>=0;--k)
      for(j=k;j<n;j++){
         temp=fabs(ck[j]);
         ck[j]= ck[j]-z[k]*ck[j+1];
         /* MM STORES THE FORWARD ERROR ESTIMATES*/
         MM[j]=MM[j]+fabs(z[k])*MM[j+1]+temp+fabs(z[k]*ck[j+1]);
	  }
   *err=0.0;
   for(i=0;i<n;i++)
    if((temp1=MM[i]*DBL_EPSILON) >= *err) *err = temp1;

   /*FREE THE ALLOCATED SPACE*/
   free(z);
   free(y1);
   free(MM);
   free(DD);
   return 0;
}
/*END OF CK_CALC()*/


   /*-------------------------------------------------------------------------*/
   /*                  COMPUTATION OF THE DIVIDED DIFFERENCES                 */
   /*                  AND  ERROR ESTIMATES  OF THE COEFFICIENTS              */
   /*-------------------------------------------------------------------------*/

int diffdiv(int n, double y1[], double z[], double DD[]) {
   int		i,j,k;
   double	num,div;

   for(k=0;k<n;k++){
       for(j=n-1;j>k;j--){
		   num=y1[j]-y1[j-1];
		   div=z[j]-z[j-k-1];
		   y1[j]=num/div;
		   DD[j]=(DD[j]-DD[j-1])/fabs(div);
	   }
	   for(i=n-1;i>k;i--) DD[i]=DD[i]+3*y1[i];
	}
	return 0;
}
