#include "dbL.h"

  /***************************************************************************\
   *                FUNCTION GZ OF THE ReLaTIve SOFTWARE PACKAGE               *
   *                          FOR NUMERICAL INVERSION	                       *
   *                       OF THE LAPLACE TRANSFORM ON                         *
   *                               THE REAL AXIS                               *
   *                                                                           *
   *               DATABASE OF INVERSE LAPLACE  TEST FUNCTIONS                 *
   *                                                                           *
   *  Authors: A. MURLI  UNIVERSITY OF NAPLES FEDERICO II                      *
   *           L. D'AMORE UNIVERSITY OF NAPLES FEDERICO II                     *
   *           R. CAMPAGNA UNIVERSITY OF NAPLES FEDERICO II                    *
   *		   V. MELE   UNIVERSITY OF NAPLES FEDERICO II                      *
   *                                                                           *
   \***************************************************************************/
   
/* G(t) = 1 */ 
double gz1(double t){
    return 1;
}

/*==================================================================*/
/* G(t) =  t */  
double gz2(double t ) {
   return t; 
}  

/*==================================================================*/

/* G(t) =  0.5*exp(-0.5*t) */ 
double gz3(double t ) { 
   return 0.5*exp(-0.5*t); 
}

/*==================================================================*/

/* G(t) = t*exp(-2*t) */
double gz4(double t ){   
   return t*exp(-2*t); 
}	

/*==================================================================*/

/* G(t) =  (1+t)*exp(t) */
double gz5(double t ) { 
   return (1+t)*exp(t); 
}

/*==================================================================*/
 
/* G(t) =  exp(2t) */ 
double gz6(double t) {
   return exp(2*t); 
}

/*==================================================================*/
	
/* G(t) = t^(n-1)/((n-1)!exp(a*t))    a=3/5 */
double gz7(double t){
    int n=5;
    double a1=3./5.;
    return gsl_sf_pow_int(t,n-1)/(gsl_sf_fact(n-1)*exp(a1*t));
}

/*==================================================================*/

 /* G(t) = (exp(-at)-exp(-bt))/(b-a)    a=3/5, b=5/7  */
 double gz8(double t){
     double a1=3./5.;
     double b1=5./7.;
     double num=(exp(-a1*t)-exp(-b1*t));
     double den=b1-a1;
     return num/den;
}

/*==================================================================*/

 /* G(t) = ((a/exp(at))-(b/exp(bt)))/(a-b)    a=3/5, b=5/7 */
 double gz9(double t){
     double a1=3./5.;
     double b1=5./7.;
     return ((a1/exp(a1*t))-(b1/exp(b1*t)))/(a1-b1);
}

/*==================================================================*/

 /* G(t) = t*cos(t) */
 double gz10(double t ){
     return t*cos(t);
} 

/*==================================================================*/
	
 /* G(t) = cos(at)    a=3/5 */
double gz11(double t){
	double a1=3./5.;
	return cos(a1*t);
}

/*==================================================================*/

/* G(t) = sin(t) */
double gz12(double t ) {
	return sin(t);
}


/*==================================================================*/

/* G(t) = 2/sqrt(3)*[exp(-x/2)*sin(x*sqrt(3)/2)] */  
double gz13(double t ) {
	return 2/sqrt(3)*(exp(-t/2)*sin(t*sqrt(3)/2)); 
} 

/*==================================================================*/

/* G(t) =  exp(-2.5*t)*sin(t) */ 
double gz14(double t ) {  
	return exp(-2.5*t)*sin(t); 
}

/*==================================================================*/

/* G(t) = (1-cos(at))/a^2    a=3/5 */
double gz15(double t){
	double a1=3./5.;
      return (1-cos(a1*t))/gsl_sf_pow_int(a1,2);
}

/*==================================================================*/

/* G(t) = (at-sin(at))/a^3    a=3/5 */
double gz16(double t){
	double a1=3./5.;
      return (a1*t-sin(a1*t))/gsl_sf_pow_int(a1,3);
}

/*==================================================================*/

/* G(t) = (sin(at)-atcos(at))/(2*a^3)    a=3/5  */
double gz17(double t){
		double a1=3./5.;
      return (sin(a1*t)-a1*t*cos(a1*t))/(2*gsl_sf_pow_int(a1,3));
}

/*==================================================================*/

/* G(t) = (sin(at)+atcos(at))/(2*a)    a=3/5 */
double gz18(double t){
	double a1=3./5.;
      return (sin(a1*t)+a1*t*cos(a1*t))/(2*a1);
}

/*==================================================================*/
	
/* G(t) = (1+a^2*t^2)*sin(at)-atcos(at)    a=3/5, b=5/7 */
double gz19(double t){
	double a1=3./5.;
  	return (1+gsl_sf_pow_int(a1,2)*gsl_sf_pow_int(t,2))*sin(a1*t)-a1*t*cos(a1*t);
}

/*==================================================================*/
   
/* G(t) = (cos(at)-cos(bt))/(b^2-a^2)    a=3/5, b=5/7 */
double gz20(double t){
		double a1=3./5.;
		double b1=5./7.;
		return (cos(a1*t)-cos(b1*t))/(gsl_sf_pow_int(b1,2)-gsl_sf_pow_int(a1,2));
}

/*==================================================================*/
	
/* G(t) =  cos(bt)/exp(at)   a=3/5, b=5/7 */
double gz21(double t){
		double a1=3./5.;
		double b1=5./7.;
		return cos(b1*t)*exp(-a1*t);
}

/*==================================================================*/

/* G(t) = exp(-at)-exp(a*t/2)*(cos(1/2*sqrt(3)*a*t)-sqrt(3)sin(1/2*sqrt(3)*a*t))    a=3/5, b=5/7 */
double gz22(double t){
		double a1=3./5.;
		return exp(-a1*t)-exp((a1*t)/2)*(cos(1./2.*sqrt(3)*a1*t)-sqrt(3)*sin(1./2.*sqrt(3)*a1*t));
}


/*==================================================================*/

/* G(t) =  (1/a)*sinh(a*t)-t  a=0.5*/
double gz23(double t ) {
		return 2*sinh(0.5*t)-t;
}

/*==================================================================*/

/* G(t) = sinh(a*t)/a    a=3/5 */
double gz24(double t){
		double a1=3./5.;
		return sinh(a1*t)/(a1);
}

/*==================================================================*/
	
/* G(t) = cosh(at)    a=3/5 */
double gz25(double t){
	double a1=3./5.;
      return cosh(a1*t);
}

/*==================================================================*/
	
/* G(t) = sin(at)*cosh(at)-cos(at)*sinh(at)    a=3/5, b=5/7 */
double gz26(double t){
	double a1=3./5.;
    	return sin(a1*t)*cosh(a1*t)-cos(a1*t)*sinh(a1*t);
}

/*==================================================================*/
   	
/* G(t) = sin(at)*sinh(at)/2*a^2    a=3/5, b=5/7 */
double gz27(double t){
	double a1=3./5.;
      return sin(a1*t)*sinh(a1*t)/(2*gsl_sf_pow_int(a1,2));
}

/*==================================================================*/
	
/* G(t) = sinh(at)-sin(at)/2*a^3    a=3/5, b=5/7 */
double gz28(double t){
	double a1=3./5.;
      return (sinh(a1*t)-sin(a1*t))/(2*gsl_sf_pow_int(a1,3));
}

/*==================================================================*/

/* G(t) = cosh(at)-cos(at)/2*a^2    a=3/5, b=5/7 */
double gz29(double t){
	double a1=3./5.;
   	return (cosh(a1*t)-cos(a1*t))/(2*gsl_sf_pow_int(a1,2));
}

/*==================================================================*/
   	
/* G(t) = (exp(-b*t)-exp(-a*t))/t   a=3/5, b=5/7 */
double gz30(double t){
	double a1=3./5.;
	double b1=5./7.;
      if(t==0) return 1;
	return (exp(-b1*t)-exp(-a1*t))/t;
}

/*==================================================================*/

  /* G(t) = (2*(1-cos(a*t)))/t   a=3/5, b=5/7 */
double gz31(double t){
	double a1=3./5.;
    	return (2*(1-cos(a1*t)))/t;
}

/*==================================================================*/

/*G(t) = (2/t)*sinh(0.5*t) */ 
double gz32(double t ) { 
  	if(t==0) return 1;
  	return 2*sinh(0.5*t)/t; 
}


/*==================================================================*/

/* G(t) = (exp(a^2 t)*Erf(a*sqrt(t)))/a, a=3/5      */
double gz33(double t){
	double a1=3./5.;
	double x=a1*sqrt(t);
	double y=gsl_sf_pow_int(a1,2)*t;
	return (exp(y)*gsl_sf_erf (x))/a1;
}

/*==================================================================*/

  
/* G(t) = exp(a^2 t)*(((b*Erf(a*sqrt(t)))/a)-1) + exp(b^2 t)*Erfc(b*sqrt(t)), a=3/5, b=5/7    */
double gz34(double t){
	double a1=3./5.;
    double b1=5./7.;
	double a2=gsl_sf_pow_int(a1,2);
	double b2=gsl_sf_pow_int(b1,2);
    double x=exp(a2*t);
    double y=exp(b2*t);
    double w=a1*sqrt(t);
    double wb=b1*sqrt(t);
	double erfw=gsl_sf_erf (w);
	double erfcwb=gsl_sf_erfc (wb);
	return x*(((b1*(erfw))/a1)-1.)+y*erfcwb;
}
/*==================================================================*/

/* G(t) = (1/(exp((1/2)*(a+b)*t) )*(t*(BesselI(0,(1/2)*(a-b)*t)) + BesselI(1,(1/2)*(a-b)*t)), a=3/5, b=5/7   */
double gz35(double t){
	double a=3./5.;
	double b=5./7.;
	double amb=a-b;
	double apb=b+a;
	double alpha=1./2.;
	double x=alpha*(apb)*t;
	double arg=alpha*(amb)*t;
	return (1./exp(x))*(t*(gsl_sf_bessel_I0 (arg)+gsl_sf_bessel_I1 (arg)));
}
/*==================================================================*/

/* G(t) = BesselI(1,a*t)/(t*exp(at)), a=3/5      */
double gz36(double t){
	double a=3./5.;
	double x=a*t;
	if (t==0) return 1;
	return gsl_sf_bessel_I1 (x)/(t*exp(x));
}

/*==================================================================*/

/*G(t) = ( n* BesselI(n,(1/2)*(a-b)*t) )/(t* exp((1/2)*(b+a)*t) ), n=5, a=3/5, b=5/7   */

double gz37(double t){                                                                                                      	
	double n=5;
	double a=3./5.;
	double b=5./7.;
	double amb=a-b;
	double apb=b+a;
	double alpha=1./2.;
	double x=alpha*(apb)*t; 
	double arg=alpha*(amb)*t;
	double outbess;
	if(t==0) return 1;	
	if(arg<0) arg=-arg;
	outbess=gsl_sf_bessel_In (n, arg);
	return (n*outbess)/(t*exp(x));
}	
	
/*==================================================================*/

/* G(t) =  BesselI(nu,a*t/2)/((a^nu)*exp(at/2)), nu=3, a=3/5  */     
double gz38(double t){
	double a=3./5.;
	double y=(a*t)/2;
	int nu=3;
	double outbess=gsl_sf_bessel_In (nu, y);
	return (outbess)/(gsl_sf_pow_int(a,nu)*exp(y));
}

/*==================================================================*/

 /* G(t) =(a^nu)* BesselI(nu,a*t), nu=3,  a=3/5  */
 double gz39(double t){
	int nu=3;
	double a=3./5.;
	double arg=a*t;
	double outbess=gsl_sf_bessel_In(nu, arg);
	return (gsl_sf_pow_int(a,nu))*outbess;
}

 /*==================================================================*/

/* G(t) = BesselJ(0,a*sqrt(t^2+2kt)), a=3/5, k=9/11  */
double gz40(double t){
	double k=9./11.;
	double a=3./5.;
	double x=2.*k*t;
	double y=gsl_sf_pow_int(t,2);
	double w=a*sqrt(y+x);
	return gsl_sf_bessel_J0 (w); 
}


	
/*==================================================================*/

/* G(t) =  a*(BesselI(1,a*t)+BesselI(0,a*t))/exp(a*t), a=3/5    */
double gz41(double t){
	double a1=3./5.;
	double a1t=a1*t;
	return a1*(gsl_sf_bessel_I1 (a1t)+gsl_sf_bessel_I0 (a1t))/exp(a1t);
}

/*==================================================================*/

/* G(t) = BesselJ(0,a*t), a=3/5    */
double gz42(double t){
    double a1=3./5.;
	double a1t=a1*t;
	return gsl_sf_bessel_J0 (a1t); 
}

/*==================================================================*/
   
/* G(t) = t^(n-1/2)/gamma(n+1/2), n=5    */
double gz43(double t){
    int n=5;
	double x=n+(1./2.);
	return pow(t,n-(1./2.))/(gsl_sf_gamma (x));
}
/*==================================================================*/

/* G(t) = 2*sqrt(t)*exp(-k^2/(4.*t))/sqrt(pi)-k*Erfc(k/2*sqrt(t)), k=9/11 */
double gz44(double t){
    double k=9./11.;
	double tmp=2.;
	double tmp2=4.;
	double x=-gsl_sf_pow_int(k,2)/(tmp2*t);
	double y=k/(tmp*sqrt(t));
	double w=tmp*sqrt(t)/sqrt(M_PI);
	return w*exp(x)-k*gsl_sf_erfc (y);
} 

   
/*==================================================================*/

/*======================================================================*/
/*
NEXT 5 are the Inverse Laplace Transform of
function TEST 1, 2, 3, 4, 5 of the paper
"On the Numerical Inversion of Laplace Transforms: Comparison of Three
New Methods on Characteristic Problems from Applications".
DEAN G. DUFFY
*/
/*======================================================================*/
/* Test 1 */
/* G(t) = (1/2) + exp(-t)*((1/2)-(exp(2)/(exp(2)-1)))-(1/pi)*sum_{n=1}^{10^7} [(sin(n*pi*t)-atan(n*pi))/(n*sqrt(n^2 * pi^2 + 1))]  */
double gz45(double t){
	double PI2=gsl_sf_pow_int(M_PI,2);
	double A=(0.5-(exp(2.0)/(exp(2.0)-1.0)))*exp(-t) + 0.5;
	double n=0;
	double sum=0;
	double B;
	double FEX;
	do{
		n=n+1.0;
		B=sin((n*M_PI*t)-atan(n*M_PI));
		B=B/(n*(sqrt((gsl_sf_pow_int(n,2)*PI2)+1.0)));
		sum=sum+B;
	}
	while (n<=1.0E+7); 
	sum=sum/M_PI;
    FEX=A-sum;
    return FEX;
}

/*==================================================================*/
/*Test 2*/

double ff(double x, void * params) {
	return x*tan(x)-1;
}

double bcalc(int i){
	int status;
    int iter = 0, max_iter = 1000000;
    const gsl_root_fsolver_type *T = gsl_root_fsolver_brent;
    gsl_root_fsolver *s = gsl_root_fsolver_alloc (T);
    double r = 0;   	
	double x_lo, x_hi; 
	gsl_function F;     
	double p=0.;

    F.function = &ff;
    F.params = &p;
	
	if(i==0){
		x_lo = 0;
		x_hi = M_PI/3.; 
	}	
	else{
		x_lo = i*M_PI;
		x_hi = (i+1)*M_PI+1./2.; 
	}
    gsl_root_fsolver_set (s, &F, x_lo, x_hi);     
    do{
		iter++;
		status = gsl_root_fsolver_iterate (s);
		r = gsl_root_fsolver_root (s);
		x_lo = gsl_root_fsolver_x_lower (s);
		x_hi = gsl_root_fsolver_x_upper (s);
		status = gsl_root_test_interval (x_lo, x_hi, 0, 0.001);
	}while (status == GSL_CONTINUE && iter < max_iter);
    gsl_root_fsolver_free (s);     
    return r;
}


double gz46(double t) { 
	double b,b2,a1,a21,a22;
	double sum=0.;
	double add=0.;
	int i=0;

	while (fabs(add)>=DBL_EPSILON*fabs(sum)){		
		sum=sum+add;
		b=bcalc(i);
		b2=gsl_pow_int(b,2);
		a1=100.+1./b2;      
		a21=2.*b*sin(b/2.)*exp(-b2*t); 
		a22=(2.+b2)*cos(b);		
		add=a1*a21/a22;
		i++;
	}
	return -1./2.+sum; 
}


/*==================================================================*/
/* Test 3 */ 
/* G(t) = (1/2) + (1/pi) integ_0^{infinity} [exp(-r*m*sqrt(u/2)*(cos(theta) - sin(theta)))*sin(t*u-r*m*sqrt(u/2)*(cos(theta)+sin(theta)))du/u],
where m=[(1+u^2)/(1+c^2 u^2)]^(1/4), 2*theta=atan(u)-atan(c*u), c=0.4, r=0.5 */

double FST3(double u, void * params) {
	double m,theta, arg1, arg2, r, c;
	double t= *(double *) params;
	double FST3val;
	r=0.5;
	c=0.4;
	m=(1.+gsl_sf_pow_int(u,2))/(1.+(gsl_sf_pow_int(c,2)*gsl_sf_pow_int(u,2)));
	m=pow(m,0.25);
	theta=atan(u)-atan(u*c);
	theta=theta/2;
	arg1=-r*m*sqrt(u/2)*(cos(theta)-sin(theta));
	arg2=t*u-(r*m*sqrt(u/2)*(cos(theta)+sin(theta)));
	FST3val=exp(arg1)*sin(arg2)/u;
	return FST3val;
}

double gz47(double t) { 
	double a=0;
	gsl_integration_workspace * w = gsl_integration_workspace_alloc(8000);
    double result, error;
    double alpha = t;
	double epsabs=0.0;
	double epsrel=1.0E-4;
	double FEX;
	gsl_function F;
    F.function = &FST3;
    F.params = &alpha;
    gsl_integration_qagiu(&F, a, epsabs, epsrel, 8000,w, &result, &error); 
    gsl_integration_workspace_free (w);
	FEX=(result/M_PI)+0.5;
	return FEX; 
}


/*==================================================================*/
/* Test 4 */ 
/* G(t) =   1- (1/pi) integ_0^{u1} [sin(ut+2k)-sin(ut-2k) du/u] + (1/pi) integ_{u2}^4 [sin(ut+2k)-sin(ut-2k) du/u], 
where  cos(k)=(1/4)*sqrt((u1^2-u^2)*(u2^2-u^2)), u1=2*sqrt(2-sqrt(3)), u2=2*sqrt(2+sqrt(3)) */
   
  double FST4(double u, void * params) { 
	double FST4val;
	double t= *(double *) params;
	double k;
	double u1=4.*(2.0-sqrt(3.0));
	double u2=4.*(2.0+sqrt(3.0));      
	double arg=(u1-gsl_sf_pow_int(u,2))*(u2-gsl_sf_pow_int(u,2));
	if (arg>= 0.0){ 
		k=acos(.25*sqrt(arg));
		FST4val=(sin(u*t+2.*k)-sin(u*t-2*k))/u;
	}
	return FST4val;
}

double gz48(double t) { 
	double FEX;
    gsl_integration_workspace * w = gsl_integration_workspace_alloc (8000);       
    double alpha = t;
	double epsabs=0.0;
	double epsrel=1.0E-4;
	double u1=2.*sqrt(2.0-sqrt(3.0));
	double u2=2.*sqrt(2.0+sqrt(3.0));
	double a=0.0;
	double b=u1;
	double result, abserr, result1, result2;     
    gsl_function F;
    F.function = &FST4;
    F.params = &alpha;    
    gsl_integration_qag(&F, a, b, epsabs, epsrel, 8000, 6, w, &result, &abserr);
	result1=result;
	a=u2;
	b=4.0;
    gsl_integration_qag(&F, a, b, epsabs, epsrel, 8000, 6, w, &result, &abserr); 
    gsl_integration_workspace_free (w);
	result2=result;
	FEX=((-result1+result2)/M_PI)+1.0;
	return FEX; 
}

/*==================================================================*/ 
/* Test 5 */ 
/* G(t) = (2/pi) integ_0^{c} [cosh(tu)(u*sqrt((R+u)/2)+ sqrt(c^2-u^2)*sqrt((R-u)/2))/(R* sqrt(c^2-u^2)*sqrt(u)) du ]+ 
(2/pi) integ_0^{b} [ (u-sqrt(c^2+u^2))/(sqrt(u)*sqrt(c^2+u^2)*sqrt(N sqrt(c^2+u^2)-u) )*cos(tu) du ], 
R=sqrt(u^2+N^2 (c^2-u^2)), b= sqrt((1-N)/(1+N)), c=(1-N)/N, N=0.5 */
   
double FST51(double u, void * params) {                  
	double t= *(double *) params;
	double FST51val;	  
	double N=0.5;
	double c=(1.-N)/N;
	double R=sqrt(gsl_sf_pow_int(u,2)+(gsl_sf_pow_int(N,2)*(gsl_sf_pow_int(c,2)-gsl_sf_pow_int(u,2)))); 
	double c2mu2=gsl_sf_pow_int(c,2)-gsl_sf_pow_int(u,2);
	FST51val=cosh(t*u)*(u*(sqrt((R+u)/2))+(sqrt(c2mu2)*sqrt((R-u)/2)))/(R*sqrt(c2mu2)*sqrt(u));   
	return FST51val;
}

double FST52(double u, void * params) {
	double t= *(double *) params;
	double FST52val;
	double N=0.5;
	double c=(1.-N)/N;
	double c2pu2=gsl_sf_pow_int(c,2)+gsl_sf_pow_int(u,2);
	FST52val=cos(t*u)*(u-sqrt(c2pu2))/(sqrt(u)*sqrt(c2pu2)*sqrt(N*sqrt(c2pu2)-u));   
	return FST52val;
}

double gz49(double t) { 
	double FEX;
    gsl_integration_workspace * w = gsl_integration_workspace_alloc(8000);     
    double result, error;
	double alpha = t;
	double epsabs=0.0;
	double epsrel=1.0E-4;
	double N=0.5;
	double b=sqrt((1.0-N)/(1.0+N));
	double c=(1.0-N)/N;
	double result1, result2;
	gsl_function G;
    gsl_function F;
    F.function = &FST51;
    F.params = &alpha;
    gsl_integration_qags(&F, 0, c, epsabs, epsrel, 8000,w, &result, &error);
	result1=result;
    G.function = &FST52;
    G.params = &alpha;
    gsl_integration_qag(&G, 0, b, epsabs, epsrel, 8000, 6, w, &result, &error);
	gsl_integration_workspace_free(w);
	result2=result;
	FEX=(result1+result2)*(2.0/M_PI);
	return FEX; 
}

  
/*==================================================================*/
