#include "dbL.h"

/* Returns the real time, in seconds */
double getRealTime( )
{
	/* AIX, BSD, Cygwin, HP-UX, Linux, OSX, POSIX, Solaris. ----- */
	struct timeval tm;
	gettimeofday( &tm, NULL );
	return tm.tv_sec + tm.tv_usec / 1000000.0;
}


/****************************************************************************                                                

THIS IS THE DRIVER PROGRAM OF  ReLaTIve:  A DOUBLE PRECISION SOFTWARE PACKAGE 
                                                   
FOR NUMERICAL INVERSION  OF A LAPLACE TRANSFORM KNOWN  ON THE REAL AXIS  WITH 

A LIMITED PRECISION 
                                                   
*******************************************************************************/

/************************************************************************\
*  Authors: 	Luisa D'Amore - University of Naples, Federico II	*
*		Rosanna Campagna - University of Naples, Federico II	*
*		Valeria Mele - University of Naples, Federico II	*
*		Almerico Murli - SPACI - NAPLES                   	*
* 		Mariarosaria Rizzardi - University of Naples Parthenope	*
\************************************************************************/

int main(int argc,char **argv){


/******************************************************************************
  
SPECIFICATION FOR VARIABLES AND CONSTANTS                                              
                                                   
********************************************************************************/

/*... Local Variables specification................................. */
/*ReLaTIve INPUT*/
	double		*x=NULL;			/*Inverse Function evaluation point(s)*/
	double		(*fz)(double);   	/*Function F Pointer*/
	int          sinf=0; 			/*optional: there's or not a singularity at infinity for F */
	int			 sflag=1;			/*user is giving the abscissa of convergence (>0) or not (<=0)*/
	
/*ReLaTIve INPUT-OUTPUT*/
	double		sigma0i=0;    		/*optional: abscissa of convergence of F */
	double 		tol=0;				/*optional: tolerance on accuracy, or 0 */
	int			nmax=0; 			/*optional: maximum number of series coefficients, or <8 */
	
/*ReLaTIve OUTPUT*/
	int			nopt;
	int			ncalc;
	double		*absesterr=NULL;	/*absolute error estimate */
	double		*relesterr=NULL;	/*relative error estimate */	
	double 		ILf=0;				/*Inverse Function f computed */
	int			*flag=NULL;			/*diagnostics on the result */
	int			ierr=0;				/*diagnostics on the software work */
	
/*AUXILIARY VARIABLES*/	
/*....ntf is the number of the functions to test together. By default it is equal to 1......*/
	int			dim=0;
	int			tf,i,j,ntf=1,ntfmax=49; 
	char		defau[5]="n",all[5]="a";
	int         range=0;
	double      a1=3./5., b1=5./7.;
	double      a=0,b=0,step=0,dimd=0.;
	double		sigma0=0.0;
	double		utol=0;
	double		g;		
/*.... true absolute error ............................ */
	double		*abserr=NULL;				
/*... true relative error...............................*/
	double		*pabserr=NULL;		
/*... Function Pointers	................................*/
	double		(*gz)(double);		
/*... Output File Pointer and Name .....................*/
	FILE		*fp;
	char		name[20]="";
/*... Timing variables .................................*/
	/*struct 		timeval startsingle, endsingle;*/
	double		elapsedsingle, elapsedtotal,maxsingle,maxsingleold,xmax=-1;
	double		tstart,tend;

	
/*****************************************************************************************************
  
THE TEST_DRIVER MAIN PROGRAM IS DESIGNED IN ORDER TO MANAGE THE TEST OF THE RELATIVE FUNCTION ON THE DATABASE FUNCTIONS

IT CAN MANAGE DIFFERENT TESTS: 

   NO ARGUMENTS :   DEFAULT TEST

                    tol= depends on the function        (DEFAULT TOLERANCE)
					
					ntf= 1				  				(DEFAULT NUMBER OF TEST FUNCTION)
					
					nmax= ReLaTIve default (40)	    (DEFAULT MAXIMUM NUMBER OF SERIES COFFICIENTS)

                    x= 1, 5, 10, 15       				(DEFAULT x-VALUES)

   1 ARGUMENT:		tol      			 				(tol could be a string (defau) to give it the default value)
   
					ntf= 1				 				(DEFAULT NUMBER OF TEST FUNCTION)
					
					nmax= ReLaTIve default (40)	    (DEFAULT MAXIMUM NUMBER OF SERIES COFFICIENTS)

                    x= 1, 5, 10, 15       				(DEFAULT x-VALUES)

   
   2 ARGUMENTS:     tol,ntf								(each one could be a string (defau) to give it the default value, ntf could be "a" to mean all the database functions)

					nmax= ReLaTIve default (40)	    (DEFAULT MAXIMUM NUMBER OF SERIES COFFICIENTS)

                    x= 1, 5, 10, 15       				(DEFAULT x-VALUES)

   3 ARGUMENTS:     tol,ntf,nmax						(each one could be a string (defau) to give it the default value, ntf could be "a" to mean all the database functions)

                    x= 1, 5, 10, 15       				(DEFAULT x-VALUES)
   
   >6 ARGUMENTS:  	tol,ntf,nmax						(each one could be a string (defau) to give it the default value, ntf could be "a" to mean all the database functions)
			
				    range								(range should be 0 or not)
            
                    IF range>0 THE INVERSE FUNCTION IS COMPUTED ON A SET OF EQUISPACED POINTS BELONGING TO 

                               THE INTERVAL [a,b] WITH STEP SIZE step. IN THIS CASE 
                       
                                5-TH ARGUMENT: a,

                                6-TH ARGUMENT: b,

                                7-TH ARGUMENT: step

					IF range=0 THE INVESRE FUNCTION IS COMPUTED ON A GVEN SET OF  VALUES. IN THIS CASE 

                                5-TH ARGUMENT: dim = NUMBER OF x VALUES 

                                6-TH TO (5+dim)-TH ARGUMENT: THE x VALUES. 

*******************************************************************************************************/
	
	printf("\n++++++++++++++++++++++++++++++\n");
	printf("   DRIVER PROGRAM STARTED!      \n");
	printf("++++++++++++++++++++++++++++++\n\n\n");

	if(argc>1){
		if (strcmp(argv[1], defau)!=0) {
			utol=atof(argv[1]);			/*first argument: tol*/
		}
		if(argc>2){
			if (strcmp(argv[2], defau)!=0){
				if (strcmp(argv[2], all)!=0)
					ntf=atoi(argv[2]);	/*second argument: ntf*/
				else ntf=ntfmax;
			}
			if(argc>3){
				if (strcmp(argv[3], defau)!=0){
					nmax=atoi(argv[3]); /*third argument: nmax*/
				}
				if(argc>4){
					range=atoi(argv[4]); /*forth argument: range*/
					if(range){				/*if range=1 then fifth, sixth, sevent arguments give interval and step for evaluation points*/
						if(argc>=8){
							a=(double)atof(argv[5]);
							b=(double)atof(argv[6]);
							step=(double)atof(argv[7]);
							dimd=(b-a)/step;
							dim=(int)(dimd+1);
							x=(double*)calloc(dim,sizeof(double));
							i=0;
							while(i<dim){
								x[i]=a;
								i++;
								a=a+step;
								
							}
						}/*endif*/
						else{
							printf("\nToo few arguments!! Please repeat the command.\n");
							printf("Note: 4th argument > 0 ----> #arguments >= 7 (>7 ignored)\n");
							exit(1);
						}/*endelse*/

					}/*endif*/
					else{
						if((argc>5)&&(strcmp(argv[5], defau)!=0)){
							dim=atoi(argv[5]);				/*if range=0 then fifth argument: number of element of x*/
							if(argc>=dim+6){
								x=(double*)calloc(dim,sizeof(double));
								for(i=0;i<dim;i++){
									x[i]=atof(argv[6+i]);		/*... and next dim arguments are the evaluation points*/
								}/*endfor*/
							}/*endif*/
							else{
								printf("\nToo few arguments!! Please repeat the command.\n");
								printf("Note: 4th argument = 0 ----> #arguments > 5 ----> #arguments >= %d (>%d ignored)\n",dim+5, dim+5);
								exit(1);
							}/*endelse*/
						}/*endif*/
						else{
							printf("\nToo few arguments!! Please repeat the command.\n");
							printf("Note: argv[4]=0 ----> #arguments > 5\n");
							exit(1);
						}/*endelse*/
					}/*endelse*/
				}/*endif*/
			}/*endif*/
		}/*endif*/
	} /*endif*/
	printf("\n--------------------------------------------------------------------");
	printf("\nuser required accuracy:\ttol=%e",utol);
	if(utol==0) printf("\n----ReLaTIve will use tol default value");
	printf("\nnumber of database functions to test:\tntf=%d",ntf);
	printf("\nuser required maximum number of coefficients:\tnmax=%d",nmax);	
	if(nmax<8) printf("\n----ReLaTIve will use nmax default value");
	if(x==NULL) {
		dim=4;
		x=(double*)calloc(dim,sizeof(double));
		x[0]=1;
		x[1]=5;
		x[2]=10;
		x[3]=15;
	}/*endif*/
	printf("\nnumber of evaluation points=\t%d\n",dim);
	printf("--------------------------------------------------------------------\n");

	
/* .............. memory space allocation of local variables..........................*/
			
	flag=(int*)calloc(dim,sizeof(int));
	relesterr=(double*)calloc(dim,sizeof(double));
	abserr=(double*)calloc(dim,sizeof(double));
	absesterr=(double*)calloc(dim,sizeof(double));
	pabserr=(double*)calloc(dim,sizeof(double));
	
	
	
/*..............TESTS START: THE OUTPUT IS PUT ON FILES NUMBERED AS THE FUNCTION NUMBER........ */

	for(j=0;j<ntf;j++){
		if(ntf<ntfmax){ printf("\nInsert the next Function to test (number):\n"); scanf("%d",&tf); }
		else tf=j+1;
		if ((tf<1)||(tf>ntfmax)){ printf("Number of function not valid! It must be in [1,%d]\n",ntfmax); j=j-1; }
		else{			
/* ......................... TEST STARTS..........................................................*/
			sinf=0;
			printf("\n************************* Function number: %d ******************************\n",tf);

			switch(tf){
			case 1:
				fz = fz1; gz = gz1; sigma0=0.; tol=1.0e-06; break; 
			case 2:
				fz = fz2; gz = gz2; sigma0=0.; tol=1.0e-06; break;
			case 3:
				fz = fz3; gz = gz3; sigma0=-0.5; tol=1.0e-06; break;
			case 4:
				fz = fz4; gz = gz4; sigma0=-2.; tol=1.0e-06; break;
			case 5:
				fz = fz5; gz = gz5; sigma0=1.; tol=1.0e-05; break; 
			case 6:
				fz = fz6; gz = gz6; sigma0=2.; tol=1.0e-02; break;
			case 7:
				fz = fz7; gz = gz7; sigma0=-a1; tol=1.0e-06; break;  
			case 8:
				fz = fz8; gz = gz8; sigma0=GSL_MAX_DBL(-a1,-b1); tol=1.0e-04; break;
			case 9:
				fz = fz9; gz = gz9; sigma0=GSL_MAX_DBL(-a1,-b1); tol=1.0e-04; break;   
			case 10:
				fz = fz10; gz = gz10; sigma0=0.; tol=1.0e-02; break;
			case 11:
				fz = fz11; gz = gz11; sigma0=0.; tol=1.0e-05; break;
			case 12:
				fz = fz12; gz = gz12; sigma0=0.; tol=1.0e-03; break;
			case 13:
				fz = fz13; gz = gz13; sigma0=0.; tol=1.0e-04; break;
			case 14:
				fz = fz14; gz = gz14; sigma0=0.; tol=1.0e-04; break;
			case 15:
				fz = fz15; gz = gz15; sigma0=0.; tol=1.0e-04; break;
			case 16:
				fz = fz16; gz = gz16; sigma0=0.; tol=1.0e-05; break;
			case 17:
				fz = fz17; gz = gz17; sigma0=0.; tol=1.0e-04; break;
			case 18:
				fz = fz18; gz = gz18; sigma0=0.; tol=1.0e-04; break;
			case 19:
				fz = fz19; gz = gz19; sigma0=0.; tol=1.0e-03; break;
			case 20:
				fz = fz20; gz = gz20; sigma0=0.; tol=1.0e-03; break;
			case 21:
				fz = fz21; gz = gz21; sigma0=0.; tol=1.0e-04; break;
			case 22:
				fz = fz22; gz = gz22; sigma0=0.; tol=1.0e-02; break;
			case 23:
				fz = fz23; gz = gz23; sigma0=0.5; tol=1.0e-05; break;
			case 24:
				fz = fz24; gz = gz24; sigma0=a1; tol=1.0e-04; break;
			case 25:
				fz = fz25; gz = gz25; sigma0=a1; tol=1.0e-04; break;
			case 26:
				fz = fz26; gz = gz26; sigma0=0.; tol=1.0e-01; break;
			case 27:
				fz = fz27; gz = gz27; sigma0=0.; tol=1.0e-01; break;
			case 28:
				fz = fz28; gz = gz28; sigma0=a1; tol=1.0e-04; break;
			case 29:
				fz = fz29; gz = gz29; sigma0=a1; tol=1.0e-04; break;
			case 30:
				fz = fz30; gz = gz30; sigma0=GSL_MAX_DBL(-a1,-b1); tol=1.0e-04; break;
			case 31:
				fz = fz31; gz = gz31; sigma0=0.; tol=1.0e-04; break;
			case 32:
				fz = fz32; gz = gz32; sigma0=0.5; tol=1.0e-05; break;
			case 33:
			       fz=fz33; gz=gz33; sigma0=gsl_sf_pow_int(a1,2); sinf=1; tol=1.0e-01; break;
		    case 34:
			       fz=fz34; gz=gz34; sigma0=GSL_MAX_DBL(gsl_sf_pow_int(GSL_MAX_DBL(-b1,0),2),gsl_sf_pow_int(a1,2)); sinf=1; tol=1.0e-01; break;
		    case 35:
		           fz=fz35; gz=gz35; sigma0=GSL_MAX_DBL(-a1,-b1); tol=1.0e-04; break;
		    case 36:
		           fz=fz36; gz=gz36; sigma0=GSL_MAX_DBL(-2*a1,0); tol=1.0e-04; break;
		    case 37:
		           fz=fz37; gz=gz37; sigma0=GSL_MAX_DBL(-a1,-b1); tol=1.0e-06; break;
		    case 38:
		           fz=fz38; gz=gz38; sigma0=GSL_MAX_DBL(-a1,0); tol=1.0e-05; break;
		    case 39:
		           fz=fz39; gz=gz39; sigma0=a1; tol=1.0e-05; break;
		    case 40:
			       fz=fz40; gz=gz40; sigma0=0; tol=1.0e-04; break;
			case 41:
				   fz=fz41; gz=gz41; sigma0=GSL_MAX_DBL(0,-2*a1); tol=1.0e-05; break;
			case 42:
					fz=fz42; gz=gz42; sigma0=0; tol=1.0e-05; break;
			case 43:
					fz=fz43; gz=gz43; sigma0=0; tol=1.0e-01; break;
			case 44:
					fz=fz44; gz=gz44; sigma0=0; tol=1.0e-01; break;
            case 45:
					fz=fz45; gz=gz45; sigma0=0; tol=1.0e-01; break;
			case 46:
					fz=fz46; gz=gz46; sigma0=0; tol=1.0e-01; break; 				
			case 47:	
					fz=fz47; gz=gz47; sigma0=0; tol=1.0e-01; break;
			case 48:	
					fz=fz48; gz=gz48; sigma0=0; tol=1.0e-01; break; 
			case 49:
					fz=fz49; gz=gz49; sigma0=1; tol=1.0e-02; break; 				
			default:
				printf("\n\n***********\nFunction number must be in [1,%d].\nWe test on a default function: 01\n***********\n\n",ntfmax);
				fz = fz1; gz = gz1; sigma0=0;  break;
			} 
			if (sinf==1){
				printf("\nThe Transform has a singularity at infinity.\n");	
			}
					
			/*... Managing output file ...............................................................*/
			sprintf(name,"fz%02dtable.txt",tf);
			printf("\nTable File: %s\n",name);
			if(!(fp=fopen(name,"w"))){ printf("File %s not open!",name); exit(1); }		
			
			fprintf(fp,"\n");
			fprintf(fp,"*******************************************************************************************************\n");
			fprintf(fp,"*                                     Test Function n.%02d                                             *\n", tf);
			fprintf(fp,"*******************************************************************************************************\n");
			fprintf(fp,"\n");
			fprintf(fp,"user required Tolerance on accuracy: %e;\n",utol);
			if(utol==0) fprintf(fp,"---- using database default tolerance for this function: %e;\n",tol);
			fprintf(fp,"user given abscissa of convergence on F: %f;\n",sigma0);
			fprintf(fp,"user required maximum number of Laguerre series coefficients: %d;\n",nmax);
			if(nmax<8) fprintf(fp,"----ReLaTIve will use nmax default value\n");

			sigma0i=sigma0;
			if(utol!=0) tol=utol;

			if (sinf==1){
				fprintf(fp,"\n*******************************************************************************************************");
				fprintf(fp,"\nRequired TRASLATION ON z in Laplace Trasform fx (sinf=%d): z-1\n",sinf);
				fprintf(fp,"TRASLATION ON sigma0 : sigma0+1=%f\n", sigma0+1);
				fprintf(fp,"DAMPING ON THE COMPUTED INVERSE: f=f*exp(-1*x)\n");
				fprintf(fp,"*******************************************************************************************************\n");
				fprintf(fp,"If sigma0<0 ReLaTIve will pose sigma0=0: here sigma0=%f\n",sigma0+1); 
			}
			else fprintf(fp,"If sigma0<0 ReLaTIve will pose sigma0=0: here sigma0=%f\n",sigma0); 
			
			fprintf(fp,"\n---------------------------------------------------------------\n");
			
			/*... timing variables initializing for the next function .....................................................*/
			elapsedsingle=0;
			elapsedtotal=0;
			maxsingle=0;
			maxsingleold=0;
			xmax=-1;
				
			/*... Evaluating The Inverse Function on all the points in x[] .................................................*/
			for(i=0; i<dim; ++i){					
				tstart= getRealTime();		
				/************************************** CALLING ReLaTIve ***************************************************/
				ierr=ReLaTIve(x[i],fz,sinf,&sigma0i,sflag,&tol,&nmax,&nopt,&ncalc,&absesterr[i],&relesterr[i],&flag[i],&ILf); 	
				/************************************************************************************************************/
				tend = getRealTime();	
				/*... Timing ..................................................*/
				elapsedsingle=tend-tstart;
				maxsingleold=maxsingle;
				maxsingle=GSL_MAX_DBL(elapsedsingle, maxsingle);
				if (maxsingle!=maxsingleold) xmax=x[i];
				elapsedtotal=elapsedtotal+elapsedsingle;

				/***********************************************************************************************************
				*                                        COMPUTED ERROR ESTIMATES                                          *
				************************************************************************************************************
				REMARK:
				AT EACH STEP OF THE DO-LOOP ReLaTIve RETURNS:
				
				as input-output:
				  1.nmax 		the used value as maximum number of coefficients to calculate
				  2.tol			the used value for the tolerance on accuracy
				  3.sigma0i		the used value for the abscissa of convergence
				  
				as output:
				  1.absesterr    ABSOLUTE ERROR ESTIMATE
				  2.relesterr    RELATIVE ERROR ESTIMATE
				  3.flag         ERROR DIAGNOSTIC
				  4.ILf          INVERSE LAPLACE FUNCTION
				  5.nopt         OPTIMAL VALUE OF n
				  5.ncalc        REACHED VALUE OF n
				  
				THE FOLLOWING WORKSPACE ARE USED:
					
				relesterr[] (double precision array)  RELATIVE ERROR ESTIMATE
									  
				pabserr[]   (double precision array)  RELATIVE ERROR
				
				absesterr[] (double precision array)  ABSOLUTE ERROR ESTIMATE
						  
				abserr[]    (double precision array)  ABSOLUTE ERROR
							
				flag[]	    (int array)               ERRORS DIAGNOSTIC
								
				**********************************************************************************************************************/
				/************************************************************************************
				 * 			           WARNING
				--------Calculated errors are just errors estimate, they are not errors bounds-------	
				
				*************************************************************************************/
					
				switch(ierr){
				case 1:
					printf("\nx=%f - ReLaTIve termined: x<0.0!\n It must be x>=0\n\n\n",x[i]);
					break;
				default: 
					if (i==0){
						fprintf(fp,"\n\nUsed Tolerance on accuracy: %e;\n",tol);
						fprintf(fp,"Used abscissa of convergence on F: %e;\n",sigma0i);
						fprintf(fp,"Used maximum number of Laguerre series coefficients: %d;\n\n",nmax);
						fprintf(fp,"\n                                                          TABLE\n");
						fprintf(fp,"---------------------------------------------------------------------------------------------------------------------------------------------------------\n");
						fprintf(fp,"|     x    |      f_comp      | True Abs Error | Estimated Abs Error | True Rel Error | Estimated Rel Error |   Nopt   |   Ncal   | FLAG | sec to calc |\n");
						fprintf(fp,"---------------------------------------------------------------------------------------------------------------------------------------------------------\n");
					}
					/*... g is the true INVERSE LAPLACE FUNCTION ...............................................................*/
					/*... Managing true errors ...............................................................*/
					g=gz(x[i]);
					if(g==0){ 
						abserr[i]=fabs(ILf)+DBL_EPSILON; 
						pabserr[i]=abserr[i];
					}
					else{
						/* abserr   absolute error       	---> trueabserr	*/
						abserr[i]=fabs(ILf-g)+DBL_EPSILON; 																
						/* pabserr	relative error       	---> truerelerr	*/
						pabserr[i]=fabs(ILf-g)/fabs(g)+DBL_EPSILON; 
						/* absesterr                     	-------> estabserr	*/
						/* relesterr                     	-------> estrelerr	*/
					}					 
					/*... Managing output file ...............................................................*/
					fprintf(fp,"| %4.2e | %16.8e | %14.3e | %19.3e | %14.3e | %19.3e | %8d | %8d | %4d | %11.3e |\n", x[i], ILf, abserr[i], absesterr[i], pabserr[i], relesterr[i], nopt, ncalc, flag[i],elapsedsingle);
					break;
				}
				/* UPDATE OF i (and x)*/
			}/*endfor on x*/
			printf("\n*****************************************************************************\n");
			fprintf(fp,"\n\n\nelapsed time (all values) in sec = %e;",elapsedtotal); 
			fprintf(fp,"\nmax elapsed time (x=%4.2e) in sec = %e;\n\n\n",xmax,maxsingle);
			print_flags_file(fp);
			print_N_file(fp,nmax);
			fclose(fp);                    
		}/*endelse (check on tf)*/               
	} /*endfor on ntf*/

	free(flag);
	free(relesterr);
	free(abserr);
	free(absesterr);
	free(pabserr);

    printf("\n\n\n++++++++++++++++++++++++++++++\n");
	printf("DRIVER PROGRAM TERMINED!\n");
	printf("++++++++++++++++++++++++++++++\n");
	/*                         END OF TESTS                                  */
	/**************************************************************************/
	
	return 0;
}
/*END OF MAIN*/

   
   
 




