!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!         Computing planar curve intersections by means of Aberth method     !
!                         by Dario A. Bini and Ana Marco                     !
!                                  v. 1.0                                    !
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!            Module : pci_mod  (parametric curve intersection)               !
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! This module contains subroutines for computing the real solutions  (t,u)   
! in the range [t1,t2]x[u1,u2] of the nonlinear algebraic  system
!                f(t,u)=0,  g(t,u)=0
! where 
!       f(t,u)=a1(t)b2(u)-a2(t)b1(u)
!       g(t,u)=a3(t)b4(u)-a4(t)b3(u)
! and a1,a2,a3,a4,b1,b2,b3,b4 are polynomials of degree na1,na2,na3,na4, 
! nb1,nb2,nb3,nb4, respectively.
! This system comes from the problem of intersecting two planar curves 
!       q1=(x1(t),y1(t)),   q2=(x2(u),y2(u)) 
! assigned in parametric form where
! x1(t)=a1(t)/a2(t), y1(t)=a3(t)/a4(t),
! x2(u)=b1(u)/b2(u), y2(t)=b3(t)/b4(t),
! and a1,a2,a3,a4,b1,b2,b3,b4 are polynomials
!
! The algorithm is based on the Ehrlic-Aberth iteration applied
! to the polynomial det(Sylv), where Sylv is the Sylvester matrix of 
! f(t,u) and g(t,u) as polynomials in u
!  ***************************************************************************
!  * All the software  contained in this library  is protected by copyright. *
!  * Permission  to use, copy, modify, and  distribute this software for any *
!  * purpose without fee is hereby granted, provided that this entire notice *
!  * is included  in all copies  of any software which is or includes a copy *
!  * or modification  of this software  and in all copies  of the supporting *
!  * documentation for such software.                                        *
!  ***************************************************************************
!  * THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR IMPLIED *
!  * WARRANTY. IN NO EVENT, NEITHER  THE AUTHORS, NOR THE PUBLISHER, NOR ANY *
!  * MEMBER  OF THE EDITORIAL BOARD OF  THE JOURNAL  "NUMERICAL ALGORITHMS", *
!  * NOR ITS EDITOR-IN-CHIEF, BE  LIABLE FOR ANY ERROR  IN THE SOFTWARE, ANY *
!  * MISUSE  OF IT  OR ANY DAMAGE ARISING OUT OF ITS USE. THE ENTIRE RISK OF *
!  * USING THE SOFTWARE LIES WITH THE PARTY DOING SO.                        *
!  ***************************************************************************
!  * ANY USE  OF THE SOFTWARE  CONSTITUTES  ACCEPTANCE  OF THE TERMS  OF THE *
!  * ABOVE STATEMENT.                                                        *
!  ***************************************************************************
!
!   CORRESPONDING AUTHOR:
!
!       DARIO ANDREA BINI
!       UNIVERSITY OF PISA, ITALY
!       E-MAIL: bini@dm.unipi.it
!
!   REFERENCE: D.A. Bini and A. Marco, "Computing curve intersection by 
!              means of simultaneous iterations"
!
! -------------------------------------------------------------------
! Main subroutine:  systemsolve
! Auxiliary subroutines: 
!     aberth
!     checkstop
!     horner
!     chorner
!     validate
!     detsyl
!     newtoncorr
!     start, cnvex, left, right, cmerge: from the package pzeros
! Auxiliary functions: bit, biton, bitoff
! -------------------------------------------------------------------
! Parameters:
! dp : double precision identifier
! nitbatch : maximum number of iterations per batch
! nit : maximum number of batch iterations
! eps : machine epsilon, used in the stop condition
! epsnewt : error bound on the Newton correction, used in the stop condition
! dolog : if .true. the program generates the file fort.200 with a log
!         about the execution steps of the program
 
MODULE pci_mod
  IMPLICIT NONE
  INTEGER,  PARAMETER :: dp=KIND(0.d0)
  INTEGER,  PARAMETER :: nit=100, nitbatch=5, threshold=20 !!D 30  !! 150
  REAL(dp), PARAMETER :: eps=EPSILON(1.d0), epsnewt=1.d-12
  LOGICAL,  PARAMETER ::  dolog=.false. !.false.

CONTAINS
  INTEGER FUNCTION bit(s,k)
!-- compute the k-th bit of the integer s
    IMPLICIT NONE
    INTEGER:: s, k
    bit=MOD(s,2**(k+1))/2**k
  END FUNCTION bit

  INTEGER FUNCTION biton(s,k)
!-- switch on the k-th bit of the integer s
    IMPLICIT NONE
    INTEGER:: s, k, ss
    ss=MOD(s,2**(k+1))/2**k
    biton=s+(1-ss)*2**k
  END FUNCTION biton

  INTEGER FUNCTION bitoff(s,k)
!-- switch off the k-th bit of the integer s
    IMPLICIT NONE
    INTEGER:: s,k,ss
    ss=MOD(s,2**(k+1))/2**k
    bitoff=s-ss*2**k
  END FUNCTION bitoff


!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!                      SUBROUTINE SYSTEMSOLVE                      !!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! ----------------------------------------------------------------------------
! INPUT variables:
! t1,t2,u1,u2: bounds for the solutions: t1<=t<=t2, u1<=u<=u2
! na1,na2,na3,na4,nb1,nb2,nb3,nb4: degrees of the polynomials
! a1,a2,a3,a4,b1,b2,b3,b4, respectively . These vectors are indexed from 0 
! and contain the polynomial coefficients ordered with increasing degree.
! OUTPUT variables:
! ub: upper bound to the number of the real solutions in the range
! t,u: complex vectors of length ub such that (t(i),u(i)), i=1:ub
!    are approximations to the solutions. 
! status: integer vector of length ub with the status of the
!    approximations. 
! The vectors t,u and status are allocated inside this subroutine
! they must not be allocated in the main program.
! The meaning of status is the following
!                Bit                                  status:
!                0 stop: small residual error            1
!                1 stop: small Newton's correction       2
!                2 spurious solution                     4
!                3 Solution at infinity                  8
!                4 nonreal solution                     16
!                5 solution in the range                32 
!                6 nonisolated solution                 64
!                7 inclusion disk disjoint from range  128
!                8 collapsed approximation             256
!                9 multiple solution                   512
!               10 infinite Newton correction         1024
!               11 numerically real                   2048
! it: number of simultaneous iterations (iterations per root)
! avit: average number of iteration per root
!--------------------------------------------------------------------
  SUBROUTINE systemsolve(t1, t2, u1, u2,                         & 
                         na1, na2, na3, na4, nb1, nb2, nb3, nb4, &
                         a1, a2, a3, a4, b1, b2, b3, b4,         &
                         ub, nsol, t, u, status, it, avit)
    IMPLICIT NONE
    INTEGER,INTENT(IN)  :: na1, na2, na3, na4, nb1, nb2, nb3, nb4
    REAL(dp),DIMENSION(0:),INTENT(INOUT) :: a1, a2, a3, a4, b1, b2, b3, b4
    REAL(dp),INTENT(INOUT)               :: t1, t2, u1, u2
    INTEGER,INTENT(OUT)                  :: ub, nsol, it 
    REAL(DP),INTENT(OUT)                 :: avit
    COMPLEX(dp),DIMENSION(:),ALLOCATABLE,INTENT(OUT) :: t, u
    INTEGER,DIMENSION(:),ALLOCATABLE,INTENT(OUT)     :: status
    
! auxiliary variables:
    INTEGER                              :: i, j, mx, my, ub1, globit, itmp
    LOGICAL                              :: same, ltmp
    COMPLEX(dp)                          :: root, s, sa1, spa1, ctmpt, ctmpu
    REAL(dp),DIMENSION(:),ALLOCATABLE    :: p, err, radt
    COMPLEX(dp),DIMENSION(:),ALLOCATABLE :: det
    LOGICAL,DIMENSION(:),ALLOCATABLE     :: cont
    REAL(dp)                             :: pi, small, big, bmxr, rtmp

      OPEN(file='systemsolve.log',unit=200)


  
!-0- check if denominators are equal  
    IF (na2==na4 .AND. nb2==nb4)THEN
       same=.FALSE.
       DO i=0,na2
          IF(a2(i)/=a4(i)) GOTO 500
       END DO
       DO i=0,nb2
          IF(b2(i)/=b4(i)) GOTO 500
       END DO
       same=.TRUE.
500    CONTINUE
    END IF

!-1- Compute an upper bound to the degree of det(Sylv)
    mx=MAX(nb1,nb2);  my=MAX(nb4,nb3)
    ub=MAX(na1+nb2,na2+nb1)*MAX(na3+nb4, na4+nb3)
    ub1=mx*MAX(na3,na4)+my*MAX(na1,na2)
    ub=MIN(ub, ub1)
    IF(same) THEN
       ub=ub-na2*MAX(nb4,nb3)
    END IF
    IF(dolog) WRITE(200,*)"same=",same, "ub=",ub

!-2- Normalize the system
    if(.not.same)then
       rtmp=max(maxval(abs(a1)), maxval(abs(a2)))
       rtmp=1.d0/sqrt(rtmp)
       a1=a1*rtmp;a2=a2*rtmp
       
       rtmp=max(maxval(abs(b1)), maxval(abs(b2)))
       rtmp=1.d0/sqrt(rtmp)
       b1=b1*rtmp;b2=b2*rtmp
    end if

!-2.1- slightly enlarge the range [t1,t2]x[u1,u2]
    t1=t1*(1-1.d-13);t2=t2*(1+1.d-13);u1=u1*(1-1.d-13);u2=u2*(1+1.d-13)


    ALLOCATE(radt(ub), status(ub), t(ub), u(ub))
    ALLOCATE(p(ub+1), det(ub+1), err(ub), cont(ub))

!-3- estimate the coefficients of det(Sylv) by means of eval/inter 
!    only if the degree is not too large
    pi=8*ATAN(1.d0)/(ub+1)

    if(ub>1000) then
       do i=1,ub
          t(i)=cos(pi*i+0.4567)+(0,1.d0)*sin(pi*i+0.4567)
       end do
       goto 10
    end if
    DO i=0, ub
       root=COS(pi*i)+SIN(pi*i)*(0,1.d0)
       CALL detsyl(na1, na2, na3, na4, nb1, nb2, nb3, nb4, mx, my, &
            a1, a2, a3, a4, b1, b2, b3, b4, root, det(i+1))
       IF(same)THEN   ! deflate the polynomial
          CALL horner(na2, root, a2, sa1, spa1)
          if(sa1/=0)then
             rtmp=abs(sa1)
             det(i+1)=det(i+1)/rtmp
             det(i+1)=det(i+1)/(sa1/rtmp**(1.d0/MAx(nb3,nb4)))**Max(nb3,nb4)
             !    det(i+1)=det(i+1)/sa1**Max(nb3,nb4)
          else
             write(*,*)"Systemsolve: Null denominator in polynomial deflation step 3"
             stop
          end if
       END IF
    END DO
 
    DO i=0,ub
       s=0 
       DO j=0,ub
          s=s+det(j+1)*(COS(pi*i*j)-(0,1.d0)*SIN(pi*i*j))
       END DO
       p(i+1)=ABS(s)/(ub+1)
    END DO

!-4- compute approximations to the roots of det(Sylv) by means
!    of Newton's polygon
    small  = TINY(1.0D0)
    big    = HUGE(1.0D0)
    IF(dolog) THEN
       write(200,*)"moduli of the polyn. coeff."
       write(200,*)p
    END IF
    CALL start(ub, p, t, radt, i, small, big)

10  continue
    IF(dolog) THEN
       write(200,*) "initial approx"
       do i=1,ub
          write(200,*)t(i)
       end do
    END IF
    ! estimate a bound to the max root
    bmxr=MAXVAL(ABS(t))*ub *0.5 
    IF(dolog) WRITE(200,*)"upper bound to the moduli:",bmxr

!-5- approximate the roots by means of Aberth's iteration
    IF(dolog) WRITE(200,*)"Call Aberth with degree=",ub
    CALL aberth(same, ub, &
         na1, na2, na3, na4, nb1, nb2, nb3, nb4,  &
         a1, a2, a3, a4, b1, b2, b3, b4, &
         bmxr, t1, t2, u1, u2, &
         t, u, cont, radt, it, status, globit)
    avit=globit*1.d0/ub
    IF(dolog) WRITE(200,*)"End of Aberth: it=",it," avit=",avit

!-6- sort the approximations and count the solutions
    j=0
    do i=1,ub
       if (bit(status(i),5)/=0 .and. bit(status(i),4)==0)then
          j=j+1
          itmp=status(i)
          rtmp=radt(i)
          ltmp=cont(i)
          ctmpt=t(i)
          ctmpu=u(i)
          status(i)=status(j)
          radt(i)=radt(j)
          cont(i)=cont(j)
          t(i)=t(j)        
          u(i)=u(j)
          status(j)=itmp
          radt(j)=rtmp
          cont(j)=ltmp
          t(j)=ctmpt
          u(j)=ctmpu
       end if
    end do
    nsol=j
    IF(dolog) THEN
       write(200,*)"solutions:",nsol
       WRITE(200,*)"CONT     STATUS    t        u       radt"
       DO i=1,ub
          If(i<=nsol)  then
             WRITE(200,*)"**", cont(i), status(i), t(i), u(i), radt(i)
          else
             WRITE(200,*) cont(i), status(i), t(i), u(i), radt(i)
          end if
       END DO
    END IF
  END SUBROUTINE systemsolve

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!                  SUBROUTINE ABERTH                    !!!!!!  
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  ! The subroutine performs the Ehrlich-Aberth iteration applied 
  ! to the equation  det(Sylv_u(p1(t,u),p2(t,u)))=0
  ! where the Newton correction newtc is computed by means of
  ! newtc =1/Sum( (Sylv_u)^{-T} * (Sylv_u)')
  ! here, * is the componentwise product and ' denotes the  
  ! derivative w.r.t. t; Sum is the sum of all the elements
  ! INPUT variables:
  !   same: true if a2=a4, b2=b4, false otherwise
  !   ub: upper bound to the number of solutions
  !   na1,..,na4, nb1,..,nb4: degrees
  !   a1,..,a4,b1,..,b4: polynomials
  !   bmxr: bound on the max root
  !   t1,t2,u1,u2: bounds on the real solutions: t1<=t<=t2, u1<=u<=u2
  ! OUTPUT variables:
  !   t,u: vectors of length ub such that (t(i),u(i)) is solution
  !   cont: vector of length ub: cont(i) is true if t(i) has not yet
  !         converged 
  ! radt: inclusion radius, there exists a solution t: |t-t(i)|<=radt(i)
  ! it, globit: number of simultaneous iterations and of global iter.
  ! status: integer vector of length ub which reports the status of t. 
  !         The bits of this integer are such that: 
  !   Bit                       status:
  !   0 stop-residual               1
  !   1 stop-newtcorr               2
  !   2 spurious solution           4
  !   3 infinite solution           8
  !   4 nonreal solution           16
  !   5 solution in the range      32 
  !   6 nonisolated solution       64
  !   7 disk disjoint from range  128
  !   8 collapsed approx.         256
  !   9 multiple sol              512
  !  10 infinite newt corr       1024
  !  11 numerically real         2048

  SUBROUTINE  aberth(same, ub,                    &
          na1, na2, na3, na4, nb1, nb2, nb3, nb4, &
          a1, a2, a3, a4, b1, b2, b3, b4,         &
          bmxr, t1, t2, u1, u2,                   &
          t, u, cont, radt, it, status, globit)
    IMPLICIT NONE
    LOGICAL, INTENT(IN) :: same
    INTEGER, INTENT(IN) :: ub, na1, na2, na3, na4, nb1, nb2, nb3, nb4
    REAL(dp), INTENT(IN)::  t1, t2, u1, u2 
    REAL(dp), INTENT(IN), DIMENSION(0:)  ::  a1, a2, a3, a4, b1, b2, b3, b4
    REAL(dp), INTENT(IN)                 :: bmxr
    COMPLEX(dp), INTENT(OUT), DIMENSION(ub):: t, u
    LOGICAL, INTENT(OUT), DIMENSION(ub)  :: cont
    REAL(dp), INTENT(OUT), DIMENSION(ub) :: radt
    INTEGER, INTENT(OUT), DIMENSION(ub)  :: status
    INTEGER, INTENT(OUT) :: it, globit

! auxiliary variables 
    INTEGER           :: mx, my, k, itbatch
    COMPLEX(dp)       :: abcorr, newtc, xnew, xx, yy
    REAL(dp)          :: pi, rcond
    INTEGER           :: info, sol
    LOGICAL           :: chkstop, disjoint
    REAL (KIND(0.d0)) :: rnd1, rnd2
    INTEGER           :: i, j, stat, nz
    REAL(KIND(0.d0))  :: res1, res2

!-1- initialize
    cont=.TRUE.
    radt=1.d100
    pi=8*ATAN(1.d0)/ub
    u=0;  rcond=1
    my=MAX(nb4,nb3); mx=MAX(nb1,nb2)
    it=0; status=16+64; globit=0

!-2- start the iterations
    DO k=1,nit
       DO itbatch=1,nitbatch
          it=it+1
          DO i=1,ub
             IF(cont(i))THEN  ! perform the iteration  
                stat=0 
                globit=globit+1

!-3- compute Newton's correction
                CALL newtoncorr(same, na1, na2, na3, na4, nb1, nb2, nb3, &
                     nb4, mx, my, a1, a2, a3, a4, b1, b2, b3, b4,     &
                     t(i), u(i), newtc, rcond, info, stat)
                radt(i)=abs(newtc)*ub*0.5

!-4- analyse exceptions
                IF(stat==1024)THEN ! infinite newton correction
                  IF(dolog) WRITE(200,*) "newtc=infty","t=",t(i)," it=",k
                   ! perform a random perturbation
                   CALL RANDOM_NUMBER(rnd1);CALL RANDOM_NUMBER(rnd2)
                   newtc=rnd1+(0,1.d0)*rnd2
                   newtc=newtc*1.d-3  !random_number*0.1d-4+(0,1)*0.123d-3
                   xnew=t(i)*(1+newtc)
                   status(i)=0
                ELSEIF(stat==4)THEN !spurious solution
                   IF(dolog) WRITE(200,*)" spurious solution",t(i)," it=",k
                   cont(i)=.FALSE.
                   status(i)=biton(status(i),2)
                   GOTO 100
                ELSEIF(stat==512)THEN !multiple solution t
                   IF(dolog) WRITE(200,*)"multiple solution",t(i),u(i)
                   cont(i)=.FALSE.
                   status(i)=biton(status(i),9)
                   GOTO 100      
                END IF

!-5- compute the Aberth correction
                abcorr=0
                DO j=1,i-1
                   IF(t(i)==t(j))THEN
                      status(i)=biton(status(i),8)
                      IF(dolog) write(200,*)"WARNING: collapsed approx.",i,t(i),status(i)
!                      write(*,*)"WARNING: collapsed approximation",t(i),status(i)
                      xnew=xnew*(1.234+(0,1.d0)*2.351)/sqrt(1.234**2+2.351**2)
                      abcorr=newtc
                      GOTO 200
                   END IF
                   abcorr=abcorr+1.d0/(t(i)-t(j))
                END DO
                DO j=i+1,ub
                   IF(t(i)==t(j))THEN
                      status(i)=biton(status(i),8)
                      IF(dolog) write(200,*)"WARNING: collapsed approx.",i,t(i),status(i)
!                      write(*,*)"collapsed approximation",t(i),status(i)
                      xnew=xnew*(1.234+(0,1.d0)*2.351)/sqrt(1.234**2+2.351**2)
                      abcorr=newtc
                      GOTO 200
                   END IF
                   abcorr=abcorr+1.d0/(t(i)-t(j))
                END DO
                abcorr=newtc/(1.d0-newtc*abcorr)
                xnew=t(i)-abcorr
200             CONTINUE
                if(abs(abcorr)<abs(t(i))*epsnewt)then
                   status(i)=biton(status(i),1)
                endif
                t(i)=xnew
100             CONTINUE
             END IF
          END DO
       END DO

!-6- Check the stop condition
       DO i=1,ub
          IF(cont(i))THEN
!  -- check for solutions at infinity          
             IF(ABS(t(i))>bmxr)THEN  !solution to infty  
                IF(dolog) WRITE(200,*) "x->infty, t=",t(i), " it=",k
                cont(i)=.FALSE.
                status(i)=biton(status(i),3)
             ELSE
                status(i)=bitoff(status(i),3)
                xx=t(i);yy=u(i)
!  -- Check if the pair (xnew,u(i)) statisfies the stop condition
                CALL checkstop(same, na1, na2, na3, na4, nb1, nb2, nb3, &
                     nb4, a1, a2, a3, a4, b1, b2, b3, b4, xx, yy, &
                     chkstop, res1, res2)
                IF(chkstop)THEN
                   cont(i)=.FALSE.
                   u(i)=yy
                   status(i)=biton(status(i),0)
! update radius
                   CALL newtoncorr(same, na1,na2,na3,na4,nb1,nb2,nb3,nb4, mx,my, &
                        a1,a2,a3,a4,b1,b2,b3,b4,t(i),u(i),newtc,& 
                        rcond,info,stat)
                   radt(i)=ub*ABS(newtc)/2
                   IF(dolog) WRITE(200,*)"it=",it,"stat=",status(i),"t=",   &
                        t(i),"u=",u(i),"res=",res1,res2,"rad=",radt(i)
                ELSE
! check for numerically real solution
                   IF(abs(aimag(t(i))).le. abs(real(t(i)))*1.d-16)then
                      status(i)=biton(status(i),11)
                      cont(i)=.false.
                   end IF
                END IF
             END IF
          END IF
       END DO

!-7- validate the approximations
          CALL validate(ub, t, u, t1, t2, u1, u2, cont, radt, status)

!-8- count roots by means of their status only if it>threshold
!    weak criterion: count if 0 or 1 or 2 or 3 or 7 or 11
       sol=0
       IF(k>threshold)THEN
!  -- update radii
          do i=1,ub
             if(cont(i))then
                CALL newtoncorr(same, na1,na2,na3,na4,nb1,nb2,nb3,nb4, mx,my, &
                     a1,a2,a3,a4,b1,b2,b3,b4,t(i),u(i),newtc,& 
                     rcond,info,stat)
                radt(i)=ub*ABS(newtc)/2
             end if
          end do
!- check for disjoint disks among cont(i)=.TRUE.
          do i=1,ub-1
             disjoint=.true.
             if(cont(i))then
                do j=i+1,ub
                   if(cont(j))then
                      if(abs(t(i)-t(j))<radt(i)+radt(j))then
                         disjoint=.false.
                         goto 2000
                      end if
                   end if
                end do
2000            continue
                if(disjoint)then
                   cont(i)=.false.
                   status(i)=bitoff(status(i),6)
                end if
             end if
          end do
! count
          sol=0
          DO i=1,ub
             if(.not.cont(i).or.status(i)==0 .or. bit(status(i),0)/=0 .or. &
                  bit(status(i),1)/=0.or.bit(status(i),2)/=0.or.      &
                  bit(status(i),3)/=0.or.(bit(status(i),7)==1) .or. bit(status(i),11)==1) then !!D
                sol=sol+1
             else 
             end if
          END DO
       end IF
          nz=0
          do i=1,ub
             if(.not. cont(i))nz=nz+1
          end do
          IF(dolog) WRITE(200,*)"Batch",k," sol=",sol,"/",ub,"nz=",nz,"/",ub
          IF(sol==ub.or.nz==ub)THEN 
             RETURN
          END IF
       IF(dolog) THEN
          WRITE(200,*)"-------------------------"
          do i=1,ub
             write(200,*)i,cont(i),status(i),t(i),radt(i)
          end do
       END IF
    END DO
    write(200,*)"Warning: reached the maximum number of iterations"
 
  END SUBROUTINE aberth


SUBROUTINE checkstop(same, na1,na2,na3,na4,nb1,nb2,nb3,nb4, &
       a1,a2,a3,a4,b1,b2,b3,b4,t,u,chkstop,ares1,ares2)
! Given the approximation (t,u) to a solution of the system, refine
! the value of u and check if (t,u) is a pseudo solution
! if so, set chkstop=.true. 
! INPUT : degree and coefficients of the polynomials a1--a4, b1--b4
!         approximation to the solution (t,u)
! OUTPUT: chkstop and the moduli ares1, ares2 of the residual errors
    IMPLICIT NONE
    LOGICAL, INTENT(IN) :: same
    INTEGER, INTENT(IN) :: na1,na2,na3,na4,nb1,nb2,nb3,nb4
    REAL(dp), INTENT(IN), DIMENSION(0:):: a1,a2,a3,a4,b1,b2,b3,b4
    COMPLEX(dp), INTENT(IN) :: t
    COMPLEX(dp), INTENT(INOUT) :: u
    LOGICAL, INTENT(OUT) :: chkstop
    REAL(dp), INTENT(OUT) :: ares1,ares2
! auxiliary variables
    REAL(dp) :: asa1,asa2,asa3,asa4,asb1,asb2,asb3,asb4,eps1,eps2
    COMPLEX(dp) :: tt,uu,sa1,sa2,sa3,sa4,sb1,sb2,sb3,sb4, &
         spa1,spa2,spa3,spa4,spb1,spb2,spb3,spb4, &
         res1,res2,z1,z2
    INTEGER :: na12,na34,nb12,nb34,i
    chkstop=.FALSE.  
    tt=t
    uu=u
    !refine only u (10 steps)
    CALL horner(na1,tt,a1,sa1,spa1)
    CALL horner(na2,tt,a2,sa2,spa2)
    CALL horner(na3,tt,a3,sa3,spa3)
    IF(same) then
       sa4=sa2; spa4=spa2
    ELSE
       CALL horner(na4,tt,a4,sa4,spa4)
    END IF
    DO i=1,10
       CALL horner(nb1,uu,b1,sb1,spb1)
       CALL horner(nb2,uu,b2,sb2,spb2)
       CALL horner(nb3,uu,b3,sb3,spb3)
       IF(same) THEN
          sb4=sb2;spb4=spb2
       ELSE
          CALL horner(nb4,uu,b4,sb4,spb4)
       END IF
       ! take a linear combination of the two equations
       z1=0.1234d0 
       z2=1-z1
       uu=uu-(z1*(sa1*sb2-sa2*sb1)+z2*(sa3*sb4-sa4*sb3  ))/   &
            (z1*(sa1*spb2-sa2*spb1)+z2*(sa3*spb4-sa4*spb3))

    END DO

    na12=max(na1,na2);na34=max(na3,na4);nb12=max(nb1,nb2);nb34=max(nb3,nb4)
    eps1=20*eps*(4*(na12+nb12)+2) *0.1  !!!!D
    eps2=20*eps*(4*(na34+nb34)+2) *0.1
    CALL chorner(na1,tt,a1,sa1,asa1)
    CALL chorner(na2,tt,a2,sa2,asa2)
    CALL chorner(na3,tt,a3,sa3,asa3)
        IF(same) then
       sa4=sa2; asa4=asa2
    ELSE
       CALL chorner(na4,tt,a4,sa4,asa4)
    END IF
    CALL chorner(nb1,uu,b1,sb1,asb1)
    CALL chorner(nb2,uu,b2,sb2,asb2)
    CALL chorner(nb3,uu,b3,sb3,asb3)
        IF(same) then
       sb4=sb2; asb4=asb2
    ELSE
       CALL chorner(nb4,uu,b4,sb4,asb4)
    END IF
    res1 = sa1*sb2-sa2*sb1
    res2 = sa3*sb4-sa4*sb3
    ares1=ABS(res1)/(asa1*asb2+asa2*asb1) 
    ares2=ABS(res2)/(asa3*asb4+asa4*asb3)
    IF(ares1<eps1 .AND. ares2<eps2) THEN
       chkstop=.TRUE.
       u=uu
    ELSE 
       chkstop=.FALSE.
    END IF
  END SUBROUTINE checkstop



  SUBROUTINE horner(n,x,p,s,s1)
! Horner's rule for evaluating polynomial and derivative
    IMPLICIT NONE
    INTEGER,INTENT(IN) :: n
    REAL(dp),INTENT(IN),DIMENSION(0:)::p
    COMPLEX(dp), INTENT(IN) :: x
    COMPLEX(dp), INTENT(OUT) :: s, s1
! auxiliary variables
    INTEGER :: i
    IF(n==0)THEN
       s=p(0)
       s1=0
       RETURN
    END IF
    s=p(n)
    s1=p(n)
    DO i=n-1,1,-1
       s=s*x+p(i)
       s1=s1*x+s
    END DO
    s=s*x+p(0)
  END SUBROUTINE horner

  SUBROUTINE chorner(n,x,p,s,as1)
! Horner rule for evaluating p(x) and |p(x)|
    IMPLICIT NONE
    INTEGER,INTENT(IN) :: n
    REAL(dp),INTENT(IN),DIMENSION(0:)::p
    COMPLEX(dp), INTENT(IN) :: x
    COMPLEX(dp), INTENT(OUT) :: s
    REAL(dp), INTENT(OUT) :: as1
! auxiliary variables
    INTEGER :: i
    REAL(dp) :: ax
    IF (n==0)THEN
       s=p(0)
       as1=ABS(s)
       RETURN
    END IF
    s=p(n)
    as1=ABS(p(n))
    ax=ABS(x)
    DO i=n-1,0,-1
       s=s*x+p(i)
       as1=as1*ax+ABS(p(i))
    END DO
  END SUBROUTINE chorner


  SUBROUTINE validate(n, x, y, t1, t2, u1, u2, cont, rad, status)
! sort and classify the approximations 
    IMPLICIT NONE
    INTEGER, INTENT(IN) :: n
    COMPLEX(KIND(0.d0)), INTENT(INOUT), DIMENSION(n) :: x,y
    REAL(KIND(0.d0)), INTENT(INOUT), DIMENSION(n)    :: rad
    LOGICAL, INTENT(INOUT), DIMENSION(n)             :: cont
    REAL(KIND(0.d0)),INTENT(IN)                      :: t1,t2,u1,u2
    INTEGER, INTENT(INOUT), DIMENSION(n)             :: status
! auxiliary variables 
    INTEGER             :: i,j,itmp
    COMPLEX(KIND(0.d0)) :: tmp
    REAL(KIND(0.d0))    :: rtmp,mx,my
    LOGICAL             :: ltmp,flag
    ! sort the approximation w.r.t. the real part  
    DO i=1,n-1
       mx=x(i)
       DO j=i+1,n
          my=x(j)
          IF(my<mx)THEN
             tmp=x(j)
             x(j)=x(i)
             x(i)=tmp
             tmp=y(j)
             y(j)=y(i)
             y(i)=tmp
             mx=my
             rtmp=rad(j)           
             rad(j)=rad(i)
             rad(i)=rtmp
             ltmp=cont(j)
             cont(j)=cont(i)
             cont(i)=ltmp
             itmp=status(j)
             status(j)=status(i)
             status(i)=itmp
          END IF
       END DO
    END DO
    ! analysis
    DO i=1,n   
! Check for complex [4], numerically real [11],   range [5] [7]
       IF(MIN(ABS(x(i)-t1),ABS(X(i)-t2))>1.5*rad(i).AND.         &
            (ABS(AIMAG(x(i)))>1.5*rad(i) .OR. t1>REAL(x(i)) .OR. & 
            REAL(x(i))>t2 ))THEN
          status(i)=biton(status(i),7)
       ELSE
          status(i)=bitoff(status(i),7)
       END IF
       !  [11]   !!D
       IF(ABS(aimag(x(i))).le. abs(real(x(i)))*1.d-16)then
          status(i)=biton(status(i),11)
       ELSE
          status(i)=bitoff(status(i),11)
       END IF

       IF(ABS(AIMAG(x(i)))>1.5*rad(i).or.                        &
            10000*abs(aimag(x(i)))>abs(real(x(i))))THEN  ! complex   !!!D 100
          status(i)=biton(status(i),4)
          status(i)=bitoff(status(i),5)
       ELSE                                            ! real
          if(abs(real(x(i)))>abs(aimag(x(i)))*1.d-8)then
             status(i)=bitoff(status(i),4)
             IF(t1<=REAL(x(i)).AND.REAL(x(i))<=t2.AND.           &
                  u1<=REAL(y(i)).AND.REAL(y(i))<=u2)THEN
                status(i)=biton(status(i),5)
             ELSE
                status(i)=bitoff(status(i),5)
             END IF
          end if
       END IF
    END DO

!  check for isolation [6] of nonspurious and finite
    DO i=1,n
       IF (bit(status(i),2)==0 .AND. bit(status(i),3)==0)THEN
          flag=.false.
          DO j=1,n
             IF(i/=j.and.bit(status(j),2)==0.AND. bit(status(j),3)==0)THEN
                IF( ABS(x(i)-x(j))<rad(i)+rad(j))THEN
                   status(j)=biton(status(j),6)
                   flag=.true.
                   goto 600
                ENDIF
             END IF
          END DO
600       continue
          IF(flag) THEN 
             status(i)=biton(status(i),6)
          ELSE
             status(i)=bitoff(status(i),6)
          END IF
       END IF
    END DO
  END SUBROUTINE validate


  SUBROUTINE detsyl(na1,na2,na3,na4,nb1,nb2,nb3,nb4, mx,my, &
       a1,a2,a3,a4,b1,b2,b3,b4, x,det)
    USE f95_lapack !, ONLY: la_getrf, la_getri
! Compute the determinant of the Sylvester matrix
    IMPLICIT NONE
    INTEGER, INTENT(IN)  :: na1,na2,na3,na4,nb1,nb2,nb3,nb4,mx,my
    REAL(dp), INTENT(IN), DIMENSION(0:) :: a1,a2,a3,a4,b1,b2,b3,b4
    COMPLEX(dp), INTENT(IN)             :: x
    COMPLEX(dp), INTENT(OUT)            :: det    
! auxiliary variables
    COMPLEX(dp),DIMENSION(mx+my,mx+my)  :: sylv
    INTEGER  ,DIMENSION(mx+my)                  :: ipiv
    REAL(dp)                            :: rcond
    COMPLEX(dp)                         :: polaux
    CHARACTER(len=1)                            :: norm,stst
    COMPLEX(dp)    :: sa1,sa3,sa2,sa4,spa1,spa2,spa3,spa4,s,s1,newtc
    INTEGER :: i, j, n, info
    norm="1"
    ! build up the Sylvester matrix and its derivative
    sylv=0
    ! first my rows
    CALL horner(na1,x,a1,sa1,spa1)
    CALL horner(na2,x,a2,sa2,spa2)
    DO j=0,MIN(nb2,nb1)
       polaux=sa1*b2(j)-sa2*b1(j)
       DO i=1,my
          sylv(i,i+j)=polaux
       END DO
    END DO

    DO j=1+MIN(nb2,nb1),MAX(nb2,nb1)
       IF(nb1>nb2)THEN
          polaux=-sa2*b1(j)
       ELSE
          polaux=sa1*b2(j)
       ENDIF
       DO i=1,my
          sylv(i,i+j)=polaux
       END DO
    END DO
    ! second mx rows
    CALL horner(na3,x,a3,sa3,spa3)
    CALL horner(na4,x,a4,sa4,spa4)
    DO j=0,MIN(nb3,nb4)
       polaux=sa3*b4(j)-sa4*b3(j)
       DO i=1,mx
          sylv(i+my,i+j)=polaux
       END DO
    END DO
    DO j=1+MIN(nb3,nb4),MAX(nb3,nb4)
       IF(nb3>nb4)THEN
          polaux=-sa4*b3(j)
       ELSE
          polaux=sa3*b4(j)
       ENDIF
       DO i=1,mx
          sylv(i+my,i+j)=polaux
       END DO
    END DO
    ! LU factorization 
    n=mx+my
    CALL la_getrf(sylv,ipiv,rcond,norm,info)
    IF(info/=0)THEN
       IF(dolog) WRITE(200,*)"Det: info=",info,"x=",x
       det=0
       RETURN
    END IF
    det=sylv(1,1)
    IF(ipiv(1)/=1)det=-det
    DO i=2,n
       det=det*sylv(i,i)
       IF(ipiv(i)/=i)det=-det
    END DO
  END SUBROUTINE detsyl




  SUBROUTINE newtoncorr(same, na1, na2, na3, na4, nb1, nb2, nb3, nb4, mx,my, &
       a1,a2,a3,a4,b1,b2,b3,b4, x, y, newtc, rcond, info, stat)
! Compute the Newton correction
!    USE la_precision , ONLY: wp => dp
    USE f95_lapack! , ONLY: la_getrf, la_getri
    IMPLICIT NONE
    LOGICAL, INTENT(IN)  :: same
    INTEGER, INTENT(IN)  :: na1,na2,na3,na4,nb1,nb2,nb3,nb4,mx,my
    INTEGER, INTENT(OUT) :: info, stat
    REAL(dp),INTENT(IN), DIMENSION(0:) :: a1,a2,a3,a4,b1,b2,b3,b4
    COMPLEX(dp), INTENT(INOUT)            :: x, y
    COMPLEX(dp), INTENT(OUT)           :: newtc  
    REAL(dp), INTENT(OUT)              :: rcond   
! auxiliary variables       
    COMPLEX(dp),DIMENSION(mx+my,mx+my) :: sylv
    INTEGER  ,DIMENSION(mx+my)         :: ipiv
    CHARACTER(len=1)                   :: norm
    INTEGER                            :: i,j,k,n,ub,  pow
    COMPLEX(dp)                        :: sa1, sa3, sa2, sa4, spa1, &
                spa2, spa3, spa4, polaux, ss, aux, yold, nwtc, s, s1
    COMPLEX(dp):: yy(mx+my)
    norm="1"
    stat=0
! build up the Sylvester matrix and its derivative
    sylv=0
! first my rows
    CALL horner(na1,x,a1,sa1,spa1)
    CALL horner(na2,x,a2,sa2,spa2)
    DO j=0,MIN(nb1,nb2)
       polaux=sa1*b2(j)-sa2*b1(j)
       DO i=1,my
          sylv(i,i+j)=polaux
       END DO
    END DO
    DO j=1+MIN(nb1,nb2),MAX(nb1,nb2)
       IF(nb1>nb2)THEN
          polaux=-sa2*b1(j)
       ELSE
          polaux=sa1*b2(j)
       END IF
       DO i=1,my
          sylv(i,i+j)=polaux
       END DO
    END DO
! second mx rows
    CALL horner(na3,x,a3,sa3,spa3)
    IF(same) then
       sa4=sa2;spa4=spa2
    ELSE
       CALL horner(na4,x,a4,sa4,spa4)
    END IF
    DO j=0,MIN(nb3,nb4)
       polaux=sa3*b4(j)-sa4*b3(j)
       DO i=1,mx
          sylv(i+my,i+j)=polaux
       END DO
    END DO
    DO j=1+MIN(nb3,nb4),MAX(nb3,nb4)
       IF(nb3>nb4)THEN
          polaux=-sa4*b3(j)
       ELSE
          polaux=sa3*b4(j)
       END IF
       DO i=1,mx
          sylv(i+my,i+j)=polaux
       END DO
    END DO
! invert the matrix
    n=mx+my
    CALL la_getrf(sylv,ipiv,rcond,norm,info)
! CHECK FOR SINGULARITY AND CONSISTENCY OF THE ROOT
    IF(info/=0 )THEN  !possible spurious solutions
       IF(dolog) WRITE(200,*)"info=",info,"x=",x,"n=",n
       if(info==1)then
          y=0
          newtc=0
          stat=1
          return
       end if
       If(info==n)then
          if(sylv(n-1,n)==0)then
             stat=4 ! spurious solution
             return
          else
             sylv(n,n)=1.d-16
             info=0
             goto 777
          end if
       else
          stat=512
          y=1
          newtc=0
          IF(dolog) WRITE(200,*)"Multiple values for u at t=",x 
          return
       end If
    END IF
777 continue

! compute the value of u
    yy(n)=1
    yold=-sylv(n-1,n-1)/sylv(n-1,n)
    DO i=n-1,1,-1
       y=0
       DO j=i+1,n
          y=y+sylv(i,j)*yy(j)
       END DO
       yy(i)=-y/sylv(i,i)
       IF(yy(i)==0)THEN
          y=0
          GOTO 400
       ELSE
          y=yy(i+1)/yy(i)
          IF(i<n-1)THEN
             IF(ABS(y-yold)<1.d-6*ABS(y))GOTO 400
          END IF
          yold=y
       END IF
    END DO
400 CONTINUE
    CALL la_getri(sylv,ipiv)
! compute the newton correction
    newtc=0
    DO j=0,MIN(nb1,nb2)
       aux=spa1*b2(j)-spa2*b1(j)
       ss=0
       DO i=1,my
          ss=ss+sylv(i+j,i)
       END DO
       newtc=newtc+ss*aux
    END DO
    DO j=MIN(nb1,nb2)+1,MAX(nb1,nb2)
       IF(nb1>nb2)THEN
          aux=-spa2*b1(j)
       ELSE
          aux=spa1*b2(j)
       END IF
       ss=0
       DO i=1,my
          ss=ss+sylv(i+j,i)
       END DO
       newtc=newtc+ss*aux
    END DO
    ! second part
    DO j=0,MIN(nb3,nb4)
       aux=spa3*b4(j)-spa4*b3(j)
       ss=0
       DO i=1,mx
          ss=ss+sylv(i+j,i+my)
       END DO
       newtc=newtc+ss*aux
    END DO
    DO j=MIN(nb3,nb4)+1,MAX(nb3,nb4)
       IF(nb3>nb4)THEN
          aux=-spa4*b3(j)
       ELSE
          aux=spa3*b4(j)
       END IF
       ss=0
       DO i=1,mx
          ss=ss+sylv(i+j,i+my)
       END DO
       newtc=newtc+ss*aux
    END DO
    IF(newtc==0)THEN
       stat=1024
       IF(dolog) WRITE(200,*)"newtc=infty"
       RETURN
    END IF
    newtc=1.d0/newtc
    IF(same)THEN ! same denominators: the polynomial can be deflated
       pow=Max(nb3,nb4)
        CALL horner(na2,x,a2,sa1,spa1)
        sa1=spa1/sa1
        newtc=newtc/(1.d0-newtc*sa1*pow)
     END IF
  END SUBROUTINE newtoncorr


  SUBROUTINE start(n, a, y, radius, nz, small, big)
! Subroutine from the package polzeros, D.A. Bini Numer.Algo.13:179-200, 1996.
    IMPLICIT NONE
    INTEGER,PARAMETER::dp=KIND(0.d0)
    INTEGER, INTENT(IN)            :: n
    INTEGER, INTENT(OUT)           :: nz
    LOGICAL,ALLOCATABLE            :: h(:)
    COMPLEX (dp), INTENT(OUT) :: y(:)
    REAL (dp), INTENT(IN)     :: small, big
    REAL (dp), INTENT(IN OUT) :: a(:)
    REAL (dp), INTENT(OUT)    :: radius(:)

    ! Local variables
    INTEGER                   :: i, iold, nzeros, j, jj
    REAL (dp)            :: r, th, ang, temp, xsmall, xbig
    REAL (dp), PARAMETER :: pi2 = 6.2831853071796, sigma = 0.7
    xsmall = LOG(small)
    xbig = LOG(big)
    nz = 0
    ALLOCATE(h(n+1))
    ! Compute the logarithm A(I) of the moduli of the coefficients of
    ! the polynomial and then the upper covex hull of the set (A(I),I)
    DO i = 1, n+1
       IF(a(i) /= 0) THEN
          a(i) = LOG(a(i))
       ELSE
          a(i) = -1.d30
       END IF
    END DO
    CALL cnvex(n+1, a, h)
    ! Given the upper convex hull of the set (A(I),I) compute the moduli
    ! of the starting approximations by means of Rouche's theorem
    iold = 1
    th = pi2/n
    DO i = 2, n+1
       IF (h(i)) THEN
          nzeros = i - iold
          temp = (a(iold) - a(i))/nzeros
          ! Check if the modulus is too small
          IF((temp < -xbig).AND.(temp >= xsmall))THEN
             IF(dolog) THEN
                WRITE(200,*)'WARNING:',nzeros,' ZERO(S) ARE TOO SMALL TO '
                WRITE(200,*)'REPRESENT THEIR INVERSES AS COMPLEX (KIND=dp) ::, THEY'
                WRITE(200,*)'ARE REPLACED BY SMALL NUMBERS, THE CORRESPONDING'
                WRITE(200,*)'RADII ARE SET TO -1'
             END IF
             nz = nz + nzeros
             r = 1.0D0/big
          END IF
          IF(temp < xsmall)THEN
             nz = nz + nzeros
             IF(dolog) THEN
                WRITE(200,*)'WARNING: ',nzeros,' ZERO(S) ARE TOO SMALL TO BE'
                WRITE(200,*)'REPRESENTED AS COMPLEX (KIND=dp) ::, THEY ARE SET 0'
                WRITE(200,*)'THE CORRESPONDING RADII ARE SET TO -1'
             END IF
          END IF
          ! Check if the modulus is too big
          IF(temp > xbig)THEN
             r = big
             nz = nz + nzeros
             IF(dolog) THEN
                WRITE(200,*)'WARNING: ',nzeros, ' ZERO(S) ARE TOO BIG TO BE'
                WRITE(200,*)'REPRESENTED AS COMPLEX (KIND=dp) ::,'
                WRITE(200,*)'THE CORRESPONDING RADII ARE SET TO -1'
             END IF
          END IF
          IF((temp <= xbig).AND.(temp > MAX(-xbig, xsmall)))THEN
             r = EXP(temp)
          END IF
          ! Compute NZEROS approximations equally distributed in the disk of
          ! radius R
          ang = pi2/nzeros
          DO j = iold, i-1
             jj = j-iold+1  
             IF((r <= (1.0D0/big)).OR.(r == big)) radius(j) = -1
             y(j) = r*(COS(ang*jj + th*i + sigma) + (0,1)*SIN(ang*jj + th*i + sigma))
          END DO
          iold = i
       END IF
    END DO
    RETURN
  END SUBROUTINE start
  
  SUBROUTINE cnvex(n, a, h)
! Subroutine from the package polzeros, D.A. Bini Numer. Algo. 13:179-200, 1996.
    IMPLICIT NONE
    INTEGER,PARAMETER::dp=KIND(0.d0)
    INTEGER, INTENT(IN)        :: n
    LOGICAL, INTENT(OUT)       :: h(:)
    REAL (dp), INTENT(IN) :: a(:)

    ! Local variables
    INTEGER :: i, j, k, m, nj, jc

    h(1:n) = .TRUE.

    ! compute K such that N-2 <= 2**K < N-1
    k = INT(LOG(n-2.0D0)/LOG(2.0D0))
    IF(2**(k+1) <= (n-2)) k = k+1

    ! For each M=1,2,4,8,...,2**K, consider the NJ pairs of consecutive
    ! sets made up by M+1 points having the common vertex
    ! (JC,A(JC)), where JC=M*(2*J+1)+1 and J=0,...,NJ,
    ! NJ = MAX(0, INT((N-2-M)/(M+M))).
    ! Compute the upper convex hull of their union by means of subroutine CMERGE
    m = 1
    DO i = 0, k
       nj = MAX(0, INT((n-2-m)/(m+m)))
       DO j = 0, nj
          jc = (j+j+1)*m+1
          CALL cmerge(n, a, jc, m, h)
       END DO
       m = m+m
    END DO
    RETURN
  END SUBROUTINE cnvex
  SUBROUTINE left(h, i, il)
! Subroutine from the package polzeros, D.A. Bini Numer. Algo. 13:179-200, 1996.
    IMPLICIT NONE
    INTEGER, INTENT(IN)  :: i
    INTEGER, INTENT(OUT) :: il
    LOGICAL, INTENT(IN)  :: h(:)

    DO il = i-1, 0, -1
       IF (h(il)) RETURN
    END DO
    RETURN
  END SUBROUTINE left

  SUBROUTINE right(n, h, i, ir)
! Subroutine from the package polzeros, D.A. Bini Numer. Algo. 13:179-200, 1996.
    IMPLICIT NONE
    INTEGER, INTENT(IN)  :: n, i
    INTEGER, INTENT(OUT) :: ir
    LOGICAL, INTENT(IN)  :: h(:)

    DO ir = i+1, n
       IF (h(ir)) RETURN
    END DO
    RETURN
  END SUBROUTINE right

  SUBROUTINE cmerge(n, a, i, m, h)
 ! Subroutine from the package polzeros, D.A. Bini Numer. Algo. 13:179-200, 1996.
   IMPLICIT NONE
    INTEGER,PARAMETER::dp=KIND(0.d0)
    INTEGER, INTENT(IN)        :: n, m, i
    LOGICAL, INTENT(IN OUT)    :: h(:)
    REAL (dp), INTENT(IN) :: a(:)

    ! Local variables
    INTEGER :: ir, il, irr, ill
    LOGICAL :: tstl, tstr

    ! at the left and the right of the common vertex (I,A(I)) determine
    ! the abscissae IL,IR, of the closest vertices of the upper convex
    ! hull of the left and right sets, respectively
    CALL left(h, i, il)
    CALL right(n, h, i, ir)

    ! check the convexity of the angle formed by IL,I,IR
    IF (ctest(a, il, i, ir)) THEN
       RETURN
    ELSE
       ! continue the search of a pair of vertices in the left and right
       ! sets which yield the upper convex hull
       h(i) = .FALSE.
       DO
          IF (il == (i-m)) THEN
             tstl = .TRUE.
          ELSE
             CALL left(h, il, ill)
             tstl = ctest(a, ill, il, ir)
          END IF
          IF (ir == MIN(n, i+m)) THEN
             tstr = .TRUE.
          ELSE
             CALL right(n, h, ir, irr)
             tstr = ctest(a, il, ir, irr)
          END IF
          h(il) = tstl
          h(ir) = tstr
          IF (tstl.AND.tstr) RETURN
          IF(.NOT.tstl) il = ill
          IF(.NOT.tstr) ir = irr
       END DO
    END IF

    RETURN
  END SUBROUTINE cmerge


  FUNCTION ctest(a, il, i, ir) RESULT(OK)
! Subroutine from the package polzeros, D.A. Bini Numer. Algo. 13:179-200, 1996.
! modified March 2006
    IMPLICIT NONE
    INTEGER, INTENT(IN)        :: i, il, ir
    REAL (dp), INTENT(IN) :: a(:)
    LOGICAL                    :: OK

    ! Local variables
    REAL (dp)            :: s1, s2
    REAL (dp), PARAMETER ::  toler =0.4

    s1 = a(i) - a(il)
    s2 = a(ir) - a(i)
    s1 = s1*(ir-i)
    s2 = s2*(i-il)
    OK = .FALSE.            
    IF(s1 > (s2+toler)) OK = .TRUE.
     RETURN
  END FUNCTION ctest

FUNCTION timing() RESULT(time)
REAL :: time
! Local variable
INTEGER :: t(8)
CALL DATE_AND_TIME(VALUES=t)
time = 3600.*t(5) + 60.*t(6) + t(7) + 0.001*t(8)
RETURN
END FUNCTION timing

END MODULE pci_mod
