function [p,R,V,U,vec] = urv_win(p,R,V,U,A,a,alg_type,tol_rank,tol_ref, ...
                                                         max_ref,fixed_rank)
%  urv_win --> Sliding window modification of the rank-revealing URV decomp.
%
%  <Synopsis>
%    [p,R,V,U,vec] = urv_win(p,R,V,U,A,a)
%    [p,R,V,U,vec] = urv_win(p,R,V,U,A,a,alg_type)
%    [p,R,V,U,vec] = urv_win(p,R,V,U,A,a,alg_type,tol_rank)
%    [p,R,V,U,vec] = urv_win(p,R,V,U,A,a,alg_type,tol_rank,tol_ref,max_ref)
%    [p,R,V,U,vec] = urv_win(p,R,V,U,A,a,alg_type,tol_rank,tol_ref, ...
%                                                       max_ref,fixed_rank)
%
%  <Description>
%    Given a rank-revealing URV decomposition of an m-by-n matrix
%    A = U*R*V' (m >= n), the function computes the updated decomposition
%    corresponding to the combined up- and down-dating action
%
%           A  ->  [ A ]  ->  [ w ]
%                  [ a ]      [ A ]
%
%    where a is a new row added to A, and w is the row that is downdated
%    after the updating process. If U is not available, then insert the
%    empty matrix [].
%
%  <Input Parameters>
%    1.   p          --> numerical rank of A revealed in R;
%    2-4. R, V, U    --> the URV factors such that A = U*R*V';
%    5.   A          --> m-by-n matrix (m >= n);
%    6.   a          --> new row added to A;
%    7.   alg_type   --> algorithm type (see Description of urv_dw);
%    8.   tol_rank   --> rank decision tolerance;
%    9.   tol_ref    --> upper bound on the 2-norm of the off-diagonal block
%                        R(1:p,p+1:n) relative to the Frobenius-norm of R;
%    10.  max_ref    --> max. number of refinement steps per singular value
%                        to achieve the upper bound tol_ref;
%    11.  fixed_rank --> if true, deflate to the fixed rank given by p
%                        instead of using the rank decision tolerance;
%
%    Defaults: alg_type = 3;
%              tol_rank = sqrt(n)*norm(R,1)*eps;
%              tol_ref  = 1e-04;
%              max_ref  = 0;
%
%  <Output Parameters>
%    1.   p       --> numerical rank of the modified A;
%    2-4. R, V, U --> the URV factors such that the modified A = U*R*V';
%    5.   vec     --> a 6-by-1 vector with:
%         vec(1) = upper bound of norm(R(1:p,p+1:n)),
%         vec(2) = estimate of pth singular value,
%         vec(3) = estimate of (p+1)th singular value,
%         vec(4) = a posteriori upper bound of num. nullspace angle,
%         vec(5) = a posteriori upper bound of num. range angle.
%         vec(6) = true if CSNE approach has been used.
%
%  <See Also>
%    ulv_win --> Sliding window modification of the rank-revealing ULV decomp.

%  <References>
%  [1] G.W. Stewart, "An Updating Algorithm for Subspace Tracking",
%      IEEE Trans. on SP, 40 (1992), pp. 1535--1541.
%
%  <Revision>
%    Ricardo D. Fierro, California State University San Marcos
%    Per Christian Hansen, IMM, Technical University of Denmark
%    Peter S.K. Hansen, IMM, Technical University of Denmark
%
%    Last revised: June 22, 1999
%-----------------------------------------------------------------------

% Check the required input arguments.
if (nargin < 6)
  error('Not enough input arguments.')
end

% Check the optional input arguments.
if (nargin < 11), fixed_rank = []; end
if (nargin < 10), max_ref    = []; end
if (nargin < 9),  tol_ref    = []; end
if (nargin < 8),  tol_rank   = []; end
if (nargin < 7),  alg_type   = []; end

% Update.
beta = 1;
[p,R,V,U,vec] = urv_up(p,R,V,U,a,beta,tol_rank,tol_ref,max_ref,fixed_rank);

% Downdate.
[p,R,V,U,vec] = urv_dw(p,R,V,U,[A; a],alg_type,tol_rank,tol_ref, ...
                                                       max_ref,fixed_rank);

%-----------------------------------------------------------------------
% End of function urv_win
%-----------------------------------------------------------------------
