function [R,V,U] = urv_ref(R,V,U,r)
%  urv_ref --> Refine one column of R in the URV decomposition.
%
%  <Synopsis>
%    [R,V,U] = urv_ref(R,V,U,r)
%
%  <Description>
%    Given the URV decomposition U*R*V', the function refines the
%    last column of R(1:r,1:r). The matrices U and V can be left out
%    by inserting an empty matrix [].
%
%  <Algorithm>
%    Refinement is an iterative algorithm, which reduces the norm of
%    the target column by applying one block QR iteration to R.
%
%  <See Also>
%    urv_cdef --> Deflate one column of R in the URV decomposition.

%  <References>
%  [1] G.W. Stewart, "An Updating Algorithm for Subspace Tracking",
%      IEEE Trans. on SP, 40 (1992), pp. 1535--1541.
%
%  [2] G.W. Stewart, "Updating a Rank-Revealing ULV Decomposition",
%      SIAM J. Matrix Anal. and Appl., 14 (1993), pp. 494--499.
%
%  <Revision>
%    Ricardo D. Fierro, California State University San Marcos
%    Per Christian Hansen, IMM, Technical University of Denmark
%    Peter S.K. Hansen, IMM, Technical University of Denmark
%
%    Last revised: June 22, 1999
%-----------------------------------------------------------------------

% Check the input arguments.
vflag = 1;
if (isempty(V)), vflag = 0; end

uflag = 1;
if (isempty(U)), uflag = 0; end

% Initialize.
[n,n] = size(R);

% Flip last column of R down.
for (i = r-1:-1:1)
  % Apply rotation to R on the right.
  [c,s,R(i,i)] = gen_giv(R(i,i),R(i,r));
  R(i,r) = 0;                               % Eliminate R(i,r).
  [R(1:i-1,i),R(1:i-1,r)] = app_giv(R(1:i-1,i),R(1:i-1,r),c,s);
  [R(r,i),R(r,r)]         = app_giv(R(r,i),R(r,r),c,s);

  % Apply rotation to V on the right.
  if (vflag)
    [V(1:n,i),V(1:n,r)] = app_giv(V(1:n,i),V(1:n,r),c,s);
  end
end

% Flip last row of R up.
for (i = 1:r-1)
  % Restore R to upper triangular form using rotation on the left.
  [c,s,R(i,i)] = gen_giv(R(i,i),R(r,i));
  R(r,i) = 0;                               % Eliminate R(r,i).
  [R(i,i+1:n),R(r,i+1:n)] = app_giv(R(i,i+1:n),R(r,i+1:n),c,s);

  % Apply rotation to U on the right.
  if (uflag)
    [U(:,i),U(:,r)] = app_giv(U(:,i),U(:,r),c,s);
  end
end

%-----------------------------------------------------------------------
% End of function urv_ref
%-----------------------------------------------------------------------
