function [R,V,U] = urv_rdef(R,V,U,r,vmax)
%  urv_rdef --> Deflate one row of R in the URV decomposition.
%
%  <Synopsis>
%    [R,V,U] = urv_rdef(R,V,U,r,vmax)
%
%  <Description>
%    Given the URV decomposition U*R*V', the function deflates
%    R(r:n,r:n). vmax is an estimate of the right singular vector
%    of R(r:n,r:n) associated with the largest singular value.
%    On return, norm(R(r,r:n)) is of the order sigma_r. The
%    matrices U and V can be left out by inserting an empty
%    matrix [].
%
%  <See Also>
%    urv_cdef --> Deflate one column of R in the URV decomposition.

%  <References>
%  [1] R.D. Fierro and P.C. Hansen, "Low-Rank Revealing UTV Decompositions",
%      Numerical Algorithms, 15 (1997), pp. 37--55.
%
%  <Revision>
%    Ricardo D. Fierro, California State University San Marcos
%    Per Christian Hansen, IMM, Technical University of Denmark
%    Peter S.K. Hansen, IMM, Technical University of Denmark
%
%    Last revised: June 22, 1999
%-----------------------------------------------------------------------

% Check the input arguments.
vflag = 1;
if (isempty(V)), vflag = 0; end

uflag = 1;
if (isempty(U)), uflag = 0; end

% Initialize.
[n,n] = size(R);

for (i = n:-1:r+1)
  % Transform vmax to e_1.
  [c,s,vmax(i-r)] = gen_giv(vmax(i-r),vmax(i-r+1));

  % Apply rotation to R on the right.
  [R(1:i,i-1),R(1:i,i)] = app_giv(R(1:i,i-1),R(1:i,i),c,s);

  % Apply rotation to V on the right.
  if (vflag)
    [V(1:n,i-1),V(1:n,i)] = app_giv(V(1:n,i-1),V(1:n,i),c,s);
  end

  % Restore R to upper triangular form using rotation on the left.
  [c,s,R(i-1,i-1)] = gen_giv(R(i-1,i-1),R(i,i-1));
  R(i,i-1) = 0;                             % Eliminate R(i,i-1).
  [R(i-1,i:n),R(i,i:n)] = app_giv(R(i-1,i:n),R(i,i:n),c,s);

  % Apply rotation to U on the right.
  if (uflag)
    [U(:,i-1),U(:,i)] = app_giv(U(:,i-1),U(:,i),c,s);
  end
end

%-----------------------------------------------------------------------
% End of function urv_rdef
%-----------------------------------------------------------------------
