function [R,V,U] = urv_cdef(R,V,U,r,vmin)
%  urv_cdef --> Deflate one column of R in the URV decomposition.
%
%  <Synopsis>
%    [R,V,U] = urv_cdef(R,V,U,r,vmin)
%
%  <Description>
%    Given the URV decomposition U*R*V', the function deflates
%    R(1:r,1:r). vmin is an estimate of the right singular vector
%    of R(1:r,1:r) associated with the smallest singular value.
%    On return, norm(R(1:r,r)) is of the order sigma_r. The
%    matrices U and V can be left out by inserting an empty
%    matrix [].
%
%  <See Also>
%    urv_rdef --> Deflate one row of R in the URV decomposition.
%    urv_ref  --> Refine one column of R in the URV decomposition.

%  <References>
%  [1] G.W. Stewart, "An Updating Algorithm for Subspace Tracking",
%      IEEE Trans. on SP, 40 (1992), pp. 1535--1541.
%
%  [2] G.W. Stewart, "Updating a Rank-Revealing ULV Decomposition",
%      SIAM J. Matrix Anal. Appl., 14 (1993), pp. 494--499.
%
%  <Revision>
%    Ricardo D. Fierro, California State University San Marcos
%    Per Christian Hansen, IMM, Technical University of Denmark
%    Peter S.K. Hansen, IMM, Technical University of Denmark
%
%    Last revised: June 22, 1999
%-----------------------------------------------------------------------

% Check the input arguments.
vflag = 1;
if (isempty(V)), vflag = 0; end

uflag = 1;
if (isempty(U)), uflag = 0; end

% Initialize.
[n,n] = size(R);

for (i = 2:r)
  % Transform vmin to e_n.
  [c,s,vmin(i)] = gen_giv(vmin(i),vmin(i-1));

  % Apply rotation to R on the right.
  [R(1:i,i-1),R(1:i,i)] = app_giv(R(1:i,i-1),R(1:i,i),c,-s);

  % Apply rotation to V on the right.
  if (vflag)
    [V(1:n,i-1),V(1:n,i)] = app_giv(V(1:n,i-1),V(1:n,i),c,-s);
  end

  % Restore R to upper triangular form using rotation on the left.
  [c,s,R(i-1,i-1)] = gen_giv(R(i-1,i-1),R(i,i-1));
  R(i,i-1) = 0;                             % Eliminate R(i,i-1).
  [R(i-1,i:n),R(i,i:n)] = app_giv(R(i-1,i:n),R(i,i:n),c,s);

  % Apply rotation to U on the right.
  if (uflag)
    [U(:,i-1),U(:,i)] = app_giv(U(:,i-1),U(:,i),c,s);
  end
end

%-----------------------------------------------------------------------
% End of function urv_cdef
%-----------------------------------------------------------------------
