function [p,L,V,U,vec] = ulv_win(p,L,V,U,A,a,alg_type,tol_rank,tol_ref, ...
                                                         max_ref,fixed_rank)
%  ulv_win --> Sliding window modification of the rank-revealing ULV decomp.
%
%  <Synopsis>
%    [p,L,V,U,vec] = ulv_win(p,L,V,U,A,a)
%    [p,L,V,U,vec] = ulv_win(p,L,V,U,A,a,alg_type)
%    [p,L,V,U,vec] = ulv_win(p,L,V,U,A,a,alg_type,tol_rank)
%    [p,L,V,U,vec] = ulv_win(p,L,V,U,A,a,alg_type,tol_rank,tol_ref,max_ref)
%    [p,L,V,U,vec] = ulv_win(p,L,V,U,A,a,alg_type,tol_rank,tol_ref, ...
%                                                       max_ref,fixed_rank)
%
%  <Description>
%    Given a rank-revealing ULV decomposition of an m-by-n matrix
%    A = U*L*V' (m >= n), the function computes the updated decomposition
%    corresponding to the combined up- and down-dating action
%
%           A  ->  [ A ]  ->  [ w ]
%                  [ a ]      [ A ]
%
%    where a is a new row added to A, and w is the row that is downdated
%    after the updating process. If U is not available, then insert the
%    empty matrix [].
%
%  <Input Parameters>
%    1.   p          --> numerical rank of A revealed in L;
%    2-4. L, V, U    --> the ULV factors such that A = U*L*V';
%    5.   A          --> m-by-n matrix (m >= n);
%    6.   a          --> new row added to A;
%    7.   alg_type   --> algorithm type (see Description of ulv_dw);
%    8.   tol_rank   --> rank decision tolerance;
%    9.   tol_ref    --> upper bound on the 2-norm of the off-diagonal block
%                        L(p+1:n,1:p) relative to the Frobenius-norm of L;
%    10.  max_ref    --> max. number of refinement steps per singular value
%                        to achieve the upper bound tol_ref;
%    11.  fixed_rank --> if true, deflate to the fixed rank given by p
%                        instead of using the rank decision tolerance;
%
%    Defaults: alg_type = 3;
%              tol_rank = sqrt(n)*norm(L,1)*eps;
%              tol_ref  = 1e-04;
%              max_ref  = 0;
%
%  <Output Parameters>
%    1.   p       --> numerical rank of the modified A;
%    2-4. L, V, U --> the ULV factors such that the modified A = U*L*V';
%    5.   vec     --> a 6-by-1 vector with:
%         vec(1) = upper bound of norm(L(p+1:n,1:p)),
%         vec(2) = estimate of pth singular value,
%         vec(3) = estimate of (p+1)th singular value,
%         vec(4) = a posteriori upper bound of num. nullspace angle,
%         vec(5) = a posteriori upper bound of num. range angle.
%         vec(6) = true if CSNE approach has been used.
%
%  <See Also>
%    urv_win --> Sliding window modification of the rank-revealing URV decomp.

%  <References>
%  [1] G.W. Stewart, "Updating a Rank-Revealing ULV Decomposition",
%      SIAM J. Matrix Anal. Appl., 14 (1993), pp. 494--499.
%
%  <Revision>
%    Ricardo D. Fierro, California State University San Marcos
%    Per Christian Hansen, IMM, Technical University of Denmark
%    Peter S.K. Hansen, IMM, Technical University of Denmark
%
%    Last revised: June 22, 1999
%-----------------------------------------------------------------------

% Check the required input arguments.
if (nargin < 6)
  error('Not enough input arguments.')
end

% Check the optional input arguments.
if (nargin < 11), fixed_rank = []; end
if (nargin < 10), max_ref    = []; end
if (nargin < 9),  tol_ref    = []; end
if (nargin < 8),  tol_rank   = []; end
if (nargin < 7),  alg_type   = []; end

% Update.
beta = 1;
[p,L,V,U,vec] = ulv_up(p,L,V,U,a,beta,tol_rank,tol_ref,max_ref,fixed_rank);

% Downdate.
[p,L,V,U,vec] = ulv_dw(p,L,V,U,[A; a],alg_type,tol_rank,tol_ref, ...
                                                       max_ref,fixed_rank);

%-----------------------------------------------------------------------
% End of function ulv_win
%-----------------------------------------------------------------------
