function [L,V,U] = ulv_ref(L,V,U,r)
%  ulv_ref --> Refine one row of L in the ULV decomposition.
%
%  <Synopsis>
%    [L,V,U] = ulv_ref(L,V,U,r)
%
%  <Description>
%    Given the ULV decomposition U*L*V', the function refines the
%    last row of L(1:r,1:r). The matrices U and V can be left out
%    by inserting an empty matrix [].
%
%  <Algorithm>
%    Refinement is an iterative algorithm, which reduces the norm of
%    the target row by applying one block QR iteration to L.
%
%  <See Also>
%    ulv_rdef --> Deflate one row of L in the ULV decomposition.

%  <References>
%  [1] G.W. Stewart, "An Updating Algorithm for Subspace Tracking",
%      IEEE Trans. on SP, 40 (1992), pp. 1535--1541.
%
%  [2] G.W. Stewart, "Updating a Rank-Revealing ULV Decomposition",
%      SIAM J. Matrix Anal. and Appl., 14 (1993), pp. 494--499.
%
%  <Revision>
%    Ricardo D. Fierro, California State University San Marcos
%    Per Christian Hansen, IMM, Technical University of Denmark
%    Peter S.K. Hansen, IMM, Technical University of Denmark
%
%    Last revised: June 22, 1999
%-----------------------------------------------------------------------

% Check the input arguments.
vflag = 1;
if (isempty(V)), vflag = 0; end

uflag = 1;
if (isempty(U)), uflag = 0; end

% Initialize.
[n,n] = size(L);

% Flip last row of L up.
for (i = r-1:-1:1)
  % Apply rotation to L on the left.
  [c,s,L(i,i)] = gen_giv(L(i,i),L(r,i));
  L(r,i) = 0;                               % Eliminate L(r,i).
  [L(i,1:i-1),L(r,1:i-1)] = app_giv(L(i,1:i-1),L(r,1:i-1),c,s);
  [L(i,r),L(r,r)]         = app_giv(L(i,r),L(r,r),c,s);

  % Apply rotation to U on the right.
  if (uflag)
    [U(:,i),U(:,r)] = app_giv(U(:,i),U(:,r),c,s);
  end
end

% Flip last column of L down.
for (i = 1:r-1)
  % Restore L to lower triangular form using rotation on the right.
  [c,s,L(i,i)] = gen_giv(L(i,i),L(i,r));
  L(i,r) = 0;                               % Eliminate L(i,r).
  [L(i+1:n,i),L(i+1:n,r)] = app_giv(L(i+1:n,i),L(i+1:n,r),c,s);

  % Apply rotation to V on the right.
  if (vflag)
    [V(1:n,i),V(1:n,r)] = app_giv(V(1:n,i),V(1:n,r),c,s);
  end
end

%-----------------------------------------------------------------------
% End of function ulv_ref
%-----------------------------------------------------------------------
