function [L,V,U] = ulv_rdef(L,V,U,r,umin)
%  ulv_rdef --> Deflate one row of L in the ULV decomposition.
%
%  <Synopsis>
%    [L,V,U] = ulv_rdef(L,V,U,r,umin)
%
%  <Description>
%    Given the ULV decomposition U*L*V', the function deflates
%    L(1:r,1:r). umin is an estimate of the left singular vector
%    of L(1:r,1:r) associated with the smallest singular value.
%    On return, norm(L(r,1:r)) is of the order sigma_r. The
%    matrices U and V can be left out by inserting an empty
%    matrix [].
%
%  <See Also>
%    ulv_cdef --> Deflate one column of L in the ULV decomposition.
%    ulv_ref  --> Refine one row of L in the ULV decomposition.

%  <References>
%  [1] G.W. Stewart, "An Updating Algorithm for Subspace Tracking",
%      IEEE Trans. on SP 40 (1992), pp. 1535--1541.
%
%  [2] G.W. Stewart, "Updating a Rank-Revealing ULV Decomposition",
%      SIAM J. Matrix Anal. Appl., 14 (1993), pp. 494--499.
%
%  <Revision>
%    Ricardo D. Fierro, California State University San Marcos
%    Per Christian Hansen, IMM, Technical University of Denmark
%    Peter S.K. Hansen, IMM, Technical University of Denmark
%
%    Last revised: June 22, 1999
%-----------------------------------------------------------------------

% Check the input arguments.
vflag = 1;
if (isempty(V)), vflag = 0; end

uflag = 1;
if (isempty(U)), uflag = 0; end

% Initialize.
[n,n] = size(L);

for (i = 2:r)
  % Transform umin to e_n.
  [c,s,umin(i)] = gen_giv(umin(i),umin(i-1));

  % Apply rotation to L on the left.
  [L(i-1,1:i),L(i,1:i)] = app_giv(L(i-1,1:i),L(i,1:i),c,-s);

  % Apply rotation to U on the right.
  if (uflag)
    [U(:,i-1),U(:,i)] = app_giv(U(:,i-1),U(:,i),c,-s);
  end

  % Restore L to lower triangular form using rotation on the right.
  [c,s,L(i-1,i-1)] = gen_giv(L(i-1,i-1),L(i-1,i));
  L(i-1,i) = 0;                             % Eliminate L(i-1,i).
  [L(i:n,i-1),L(i:n,i)] = app_giv(L(i:n,i-1),L(i:n,i),c,s);

  % Apply rotation to V on the right.
  if (vflag)
    [V(1:n,i-1),V(1:n,i)] = app_giv(V(1:n,i-1),V(1:n,i),c,s);
  end
end

%-----------------------------------------------------------------------
% End of function ulv_rdef
%-----------------------------------------------------------------------
