function [L,V,U] = ulv_cdef(L,V,U,r,umax)
%  ulv_cdef --> Deflate one column of L in the ULV decomposition.
%
%  <Synopsis>
%    [L,V,U] = ulv_cdef(L,V,U,r,umax)
%
%  <Description>
%    Given the ULV decomposition U*L*V', the function deflates
%    L(r:n,r:n). umax is an estimate of the left singular vector
%    of L(r:n,r:n) associated with the largest singular value.
%    On return, norm(L(r:n,r)) is of the order sigma_r. The
%    matrices U and V can be left out by inserting an empty
%    matrix [].
%
%  <See Also>
%    ulv_rdef --> Deflate one row of L in the ULV decomposition.

%  <References>
%  [1] R.D. Fierro and P.C. Hansen, "Low-Rank Revealing UTV Decompositions",
%      Numerical Algorithms, 15 (1997), pp. 37--55.
%
%  <Revision>
%    Ricardo D. Fierro, California State University San Marcos
%    Per Christian Hansen, IMM, Technical University of Denmark
%    Peter S.K. Hansen, IMM, Technical University of Denmark
%
%    Last revised: June 22, 1999
%-----------------------------------------------------------------------

% Check the input arguments.
vflag = 1;
if (isempty(V)), vflag = 0; end

uflag = 1;
if (isempty(U)), uflag = 0; end

% Initialize.
[n,n] = size(L);

for (i = n:-1:r+1)
  % Transform umax to e_1.
  [c,s,umax(i-r)] = gen_giv(umax(i-r),umax(i-r+1));

  % Apply rotation to L on the left.
  [L(i-1,1:i),L(i,1:i)] = app_giv(L(i-1,1:i),L(i,1:i),c,s);

  % Apply rotation to U on the right.
  if (uflag)
    [U(:,i-1),U(:,i)] = app_giv(U(:,i-1),U(:,i),c,s);
  end

  % Restore L to lower triangular form using rotation on the right.
  [c,s,L(i-1,i-1)] = gen_giv(L(i-1,i-1),L(i-1,i));
  L(i-1,i) = 0;                             % Eliminate L(i-1,i).
  [L(i:n,i-1),L(i:n,i)] = app_giv(L(i:n,i-1),L(i:n,i),c,s);

  % Apply rotation to V on the right.
  if (vflag)
    [V(1:n,i-1),V(1:n,i)] = app_giv(V(1:n,i-1),V(1:n,i),c,s);
  end
end

%-----------------------------------------------------------------------
% End of function ulv_cdef
%-----------------------------------------------------------------------
