function ullvdemo
%  ullvdemo --> Demonstrates initial and windowed ULLV algorithms.

%  <Revision>
%    Ricardo D. Fierro, California State University San Marcos
%    Per Christian Hansen, IMM, Technical University of Denmark
%    Peter S.K. Hansen, IMM, Technical University of Denmark
%
%    Last revised: January 2, 2005
%-----------------------------------------------------------------------

close all;

fprintf(1,'.                                                           \n');
fprintf(1,'.  The purpose of ULLVDEMO is to demonstrate the            \n');
fprintf(1,'.  use of the high-rank ULLV algorithm in ULLV,             \n');
fprintf(1,'.  and the up- and downdating algorithms in ULLV_UP_A       \n');
fprintf(1,'.  and ULLV_DW_A.                                           \n');
fprintf(1,'.                                                           \n');
input('.                                 [Press RETURN to continue]');
fprintf(1,'.                                                           \n');

format short e;

% Test matrix generation.
m = 50;
n = 20;

randn('seed',100);
AB_pinv = randn(m,m);

numrank = 13;

[ua,gamma,ub] = svd(AB_pinv);

gamma1 = 2*logspace(1,-3,numrank);
gamma2 = 5*logspace(-4,-6,n-numrank);
gamma  = [gamma1 gamma2]';

c = gamma./(sqrt(gamma.^2 + 1));
s = c./gamma;

randn('seed',200);
X = randn(n,n);

A = ua(:,1:n)*diag(c)*X';
B = ub(:,1:n)*diag(s)*X';

% Define input parameters.
tol_rank = 0.001;
tol_ref  = 1e-04;
max_ref  = 2;

fprintf(1,'.                                                           \n');
fprintf(1,'.  Input parameters to ULLV                                 \n');
fprintf(1,'............................................................\n');
fprintf(1,'.    no. rows m of A and B               = %3.0f \n',m);
fprintf(1,'.    no. cols n of A and B               = %3.0f \n',n);
fprintf(1,'.    rank tolerance (tol_rank)           = %6.4e \n',tol_rank);
fprintf(1,'.    refinement tolerance (tol_ref)      = %6.4e \n',tol_ref);
fprintf(1,'.    max refinement steps (max_ref)      = %3.0f \n',max_ref);
fprintf(1,'............................................................\n');
fprintf(1,'.                                                           \n');
fprintf(1,'.                                                           \n');
fprintf(1,'.  Compute the rank-revealing ULLV decomposition:           \n');
fprintf(1,'.                                                           \n');
fprintf(1,'.    [p,LA,L,V,UA,UB,vec] = ullv(A,B,tol_rank,tol_ref, ...  \n');
fprintf(1,'.                                                 max_ref)  \n');
fprintf(1,'.                                                           \n');
input('.                                 [Press RETURN to continue]');
fprintf(1,'.                                                           \n');
fprintf(1,'.  wait...                                                  \n');
fprintf(1,'.                                                           \n');

% Compute ULLV.
t0 = cputime;

[p,LA,L,V,UA,UB,vec] = ullv(A,B,tol_rank,tol_ref,max_ref);

time = cputime - t0;

lnorm = norm(LA(p+1:n,1:p));
theta = norm(UB(:,1:p)'*ub(:,p+1:m));
phi   = norm(UA(:,1:p)'*ua(:,p+1:m));

fprintf(1,'.                                                           \n');
fprintf(1,'.  Output results from ULLV                                 \n');
fprintf(1,'............................................................\n');
fprintf(1,'.    numerical rank p of A*pinv(B)       = %3.0f \n',p);
fprintf(1,'.    upper bound of ||LA(p+1:n,1:p)||    = %6.4e \n',vec(1));
fprintf(1,'.    estimate of pth singular value      = %6.4e \n',vec(2));
fprintf(1,'.    estimate of (p+1)th singular value  = %6.4e \n',vec(3));
fprintf(1,'.    estimate of num. nullspace angle    = %6.4e \n',vec(4));
fprintf(1,'.    estimate of num. range angle        = %6.4e \n',vec(5));
fprintf(1,'............................................................\n');
fprintf(1,'.                                                           \n');
fprintf(1,'.                                                           \n');
fprintf(1,'.  GSVD based results                                       \n');
fprintf(1,'............................................................\n');
fprintf(1,'.    numerical rank p of A*pinv(B)       = %3.0f \n',numrank);
fprintf(1,'.    ||LA(p+1:n,1:p)||                   = %6.4e \n',lnorm);
fprintf(1,'.    pth singular value                  = %6.4e \n',gamma1(numrank));
fprintf(1,'.    (p+1)th singular value              = %6.4e \n',gamma2(1));
fprintf(1,'.    numerical nullspace angle           = %6.4e \n',theta);
fprintf(1,'.    numerical range angle               = %6.4e \n',phi);
fprintf(1,'............................................................\n');
fprintf(1,'.                                                           \n');
fprintf(1,'.                                                           \n');
fprintf(1,'.  Computational complexity of ULLV                         \n');
fprintf(1,'............................................................\n');
fprintf(1,'.    CPU time                            = %6.4e \n',time);
fprintf(1,'............................................................\n');
fprintf(1,'.                                                           \n');
input('.                                 [Press RETURN to continue]');
fprintf(1,'.                                                           \n');
fprintf(1,'.                                                           \n');
fprintf(1,'.  Now the up- and downdating ULLV algorithms.              \n');
fprintf(1,'.                                                           \n');
input('.                                 [Press RETURN to continue]');
fprintf(1,'.                                                           \n');

% Test matrix generation.
M = 24;
M = 20;
m = 5;
n = 5;
alpha = 1e-7;

randn('seed',100);
X1 = randn(8,5);

[U,Sigma,V] = svd(X1);

Sigma(4,4) = alpha;
Sigma(5,5) = alpha;

X1 = U*Sigma*V';

randn('seed',101);
X2 = randn(8,5);

[U,Sigma,V] = svd(X2);

Sigma(5,5) = alpha;

X2 = U*Sigma*V';

randn('seed',102);
X3 = randn(8,5);

[U,Sigma,V] = svd(X3);

Sigma(5,5) = alpha;

X3 = U*Sigma*V';

X = [X1; X2; X3];

randn('seed',200);
B = randn(5,5);

% Define input parameters.
beta     = 1;
tol_rank = 1e-5;
tol_ref  = 1e-4;
max_ref  = 1;

fprintf(1,'.                                                           \n');
fprintf(1,'.  The first experiment maintains the left orthogonal       \n');
fprintf(1,'.  matrices UA and UB, and the algorithm is shown           \n');
fprintf(1,'.  to be reliable.                                          \n');
fprintf(1,'.                                                           \n');
fprintf(1,'.  Input parameters to ULLV_UP_A and ULLV_DW_A              \n');
fprintf(1,'............................................................\n');
fprintf(1,'.    no. rows m of A and B               = %3.0f \n',m);
fprintf(1,'.    no. cols n of A and B               = %3.0f \n',n);
fprintf(1,'.    no. of new rows added to A          = %3.0f \n',M-m);
fprintf(1,'.    forgetting factor (beta)            = %3.0e \n',beta);
fprintf(1,'.    rank tolerance (tol_rank)           = %6.4e \n',tol_rank);
fprintf(1,'.    refinement tolerance (tol_ref)      = %6.4e \n',tol_ref);
fprintf(1,'.    max refinement steps (max_ref)      = %3.0f \n',max_ref);
fprintf(1,'............................................................\n');
fprintf(1,'.                                                           \n');
fprintf(1,'.                                                           \n');
fprintf(1,'.  Modifying the rank-revealing ULLV decomposition (loop):  \n');
fprintf(1,'.                                                           \n');
fprintf(1,'.    [p,LA,L,V,UA,UB,vec] = ullv_up_a(p,LA,L,V,UA,UB,a, ... \n');
fprintf(1,'.                            beta,tol_rank,tol_ref,max_ref) \n');
fprintf(1,'.                                                           \n');
fprintf(1,'.    [p,LA,L,V,UA,UB,vec] = ullv_dw_a(p,LA,L,V,UA,UB,A, ... \n');
fprintf(1,'.                                 tol_rank,tol_ref,max_ref) \n');
fprintf(1,'.                                                           \n');
input('.                                 [Press RETURN to continue]');
fprintf(1,'.                                                           \n');
fprintf(1,'.  wait...                                                  \n');
fprintf(1,'.                                                           \n');

% Initialize test output.
I  = eye(n);
T1 = zeros(M,1);
T2 = zeros(M,1);
T3 = zeros(M,1);
p  = zeros(M,1);
pr = zeros(M,1);

% Handle the first matrix.
A = X(1:m,1:n);

[p(m),LA,L,V,UA,UB,vec] = ullv(A,B,tol_rank,tol_ref,max_ref);

% GSVD as reference.
Sigma = gsvd(A,B);
pr(m) = n - max([find(Sigma<tol_rank); 0]);

% Test identity.
T1(m) = norm(I - V'*V);
% Test decomposition.
T2(m) = norm(A - UA*LA*L*V');
T3(m) = norm(B - UB*L*V');

% The main loop.
for (k = m+1:M)
  fprintf(1,'.    Row no. %g \n',k);

  % New row in data matrix.
  a = X(k,:);

  [p(k),LA,L,V,UA,UB,vec] = ullv_up_a(p(k-1),LA,L,V,UA,UB,a, ...
                                      beta,tol_rank,tol_ref,max_ref);

  [p(k),LA,L,V,UA,UB,vec] = ullv_dw_a(p(k),LA,L,V,UA,UB,[A; a], ...
                                      tol_rank,tol_ref,max_ref);

  A = [A(2:m,:); a];

  % GSVD as reference.
  Sigma = gsvd(A,B);
  pr(k) = n - max([find(Sigma<tol_rank); 0]);

  % Test identity.
  T1(k) = norm(I - V'*V);
  % Test decomposition.
  T2(k) = norm(A - UA*LA*L*V');
  T3(k) = norm(B - UB*L*V');
end

figure;

subplot(3,3,1),plot(1:M,p,'x',1:M,pr,'o');
axis([m M 0 inf]);
axis 'auto y';
ylabel('Rank Estimate');
title('o=GSVD and x=ULLV');

subplot(3,3,2),plot(T1);
axis([m M 0 inf]);
axis 'auto y';
ylabel('Orthogonality Error in V');

subplot(3,3,3),plot([T2 T3]);
axis([m M 0 inf]);
axis 'auto y';
ylabel('Decomposition Errors');

fprintf(1,'.                                                           \n');
fprintf(1,'.  Output results from ULLV up- and downdating algorithms   \n');
fprintf(1,'............................................................\n');
fprintf(1,'.    fig. 1: rank estimate of A*pinv(B)                     \n');
fprintf(1,'.    fig. 2: orthogonality error = norm(I - V^T*V)          \n');
fprintf(1,'.    fig. 3: decomposition error = norm(A - UA*LA*L*V^T)    \n');
fprintf(1,'.            decomposition error = norm(B - UB*L*V^T)       \n');
fprintf(1,'............................................................\n');
fprintf(1,'.                                                           \n');
input('.                                 [Press RETURN to continue]');
fprintf(1,'.                                                           \n');
fprintf(1,'.                                                           \n');
fprintf(1,'.  Now repeat the experiment but without maintaining the    \n');
fprintf(1,'.  left orthogonal matrices UA and UB.                      \n');
fprintf(1,'.                                                           \n');
input('.                                 [Press RETURN to continue]');
fprintf(1,'.                                                           \n');
fprintf(1,'.  wait...                                                  \n');
fprintf(1,'.                                                           \n');

% Initialize test output.
I  = eye(n);
T1 = zeros(M,1);
T2 = zeros(M,1);
p  = zeros(M,1);
pr = zeros(M,1);

% Handle the first matrix.
A = X(1:m,1:n);

[p(m),LA,L,V] = ullv(A,B,tol_rank,tol_ref,max_ref);
UA = []; UB = [];

% GSVD as reference.
Sigma = gsvd(A,B);
pr(m) = n - max([find(Sigma<tol_rank); 0]);

% Test identity.
T1(m) = norm(I - V'*V);

% The main loop.
for (k = m+1:M)
  fprintf(1,'.    Row no. %g \n',k);

  % New row in data matrix.
  a = X(k,:);

  [p(k),LA,L,V,UA,UB,vec] = ullv_up_a(p(k-1),LA,L,V,UA,UB,a, ...
                                      beta,tol_rank,tol_ref,max_ref);

  [p(k),LA,L,V,UA,UB,vec] = ullv_dw_a(p(k),LA,L,V,UA,UB,[A; a], ...
                                      tol_rank,tol_ref,max_ref);

  A = [A(2:m,:); a];

  % GSVD as reference.
  Sigma = gsvd(A,B);
  pr(k) = n - max([find(Sigma<tol_rank); 0]);

  % Test identity.
  T1(k) = norm(I - V'*V);
  % Test csne/linpack flag.
  T2(k) = vec(6);
end

subplot(3,3,4),plot(1:M,p,'x',1:M,pr,'o');
axis([m M 0 inf]);
axis 'auto y';
xlabel('Index of New Row');
ylabel('Rank Estimate');

subplot(3,3,5),plot(T1);
axis([m M 0 inf]);
axis 'auto y';
xlabel('Index of New Row');
ylabel('Orthogonality Error in V');

subplot(3,3,6),plot(T2,'x');
axis([m M 0 1.2]);
xlabel('Index of New Row');
ylabel('CSNE/Linpack Flag');

fprintf(1,'.                                                           \n');
fprintf(1,'.  Output results from ULLV up- and downdating algorithms   \n');
fprintf(1,'............................................................\n');
fprintf(1,'.    fig. 4: rank estimate of A*pinv(B)                     \n');
fprintf(1,'.    fig. 5: orthogonality error = norm(I - V^T*V)          \n');
fprintf(1,'.    fig. 6: one if CSNE method has been used in downdating \n');
fprintf(1,'............................................................\n');
fprintf(1,'.                                                           \n');

%-----------------------------------------------------------------------
% End of function ullvdemo
%-----------------------------------------------------------------------
