function [LA,L,V,UA,UB] = ullv_ref(LA,L,V,UA,UB,r)
%  ullv_ref --> Refine one row of LA in the ULLV decomposition.
%
%  <Synopsis>
%    [LA,L,V,UA,UB] = ullv_ref(LA,L,V,UA,UB,r)
%
%  <Description>
%    Given the ULLV decomposition of the matrix pair A = UA*LA*L*V'
%    and B = UB*L*V', the function refines the last row of LA(1:r,1:r).
%    The matrices UA, UB and V can be left out by inserting an empty
%    matrix [].
%
%  <Algorithm>
%    Refinement is an iterative algorithm, which reduces the norm of
%    the target row by applying one block QR iteration to LA.
%
%  <See Also>
%    ullv_rdef --> Deflate one row of LA in the ULLV decomposition.

%  <References>
%  [1] S.Qiao, "Computing the ULLV Decomposition", CRL Report 278,
%      Communications Research Laboratory, McMaster Uni., Hamilton,
%      Canada, pp. 1--13, January, (1994).
%
%  [2] G.W. Stewart, "An Updating Algorithm for Subspace Tracking",
%      IEEE Trans. on SP, 40 (1992), pp. 1535--1541.
%
%  [3] G.W. Stewart, "Updating a Rank-Revealing ULV Decomposition",
%      SIAM J. Matrix Anal. and Appl., 14 (1993), pp. 494--499.
%
%  <Revision>
%    Ricardo D. Fierro, California State University San Marcos
%    Per Christian Hansen, IMM, Technical University of Denmark
%    Peter S.K. Hansen, IMM, Technical University of Denmark
%
%    Last revised: June 22, 1999
%-----------------------------------------------------------------------

% Check the input arguments.
vflag = 1;
if (isempty(V)), vflag = 0; end

uAflag = 1;
if (isempty(UA)), uAflag = 0; end

uBflag = 1;
if (isempty(UB)), uBflag = 0; end

% Initialize.
[n,n] = size(LA);

% Flip last row of LA(1:r,1:r) up.
for (i = r-1:-1:1)
  % Apply rotation to LA on the left.
  [c,s,LA(i,i)] = gen_giv(LA(i,i),LA(r,i));
  LA(r,i) = 0;                              % Eliminate LA(r,i).
  [LA(i,1:i-1),LA(r,1:i-1)] = app_giv(LA(i,1:i-1),LA(r,1:i-1),c,s);
  [LA(i,r),LA(r,r)]         = app_giv(LA(i,r),LA(r,r),c,s);

  % Apply rotation to UA on the right.
  if (uAflag)
    [UA(:,i),UA(:,r)] = app_giv(UA(:,i),UA(:,r),c,s);
  end
end

% Zero all but the first two elements of the last row of L(1:r,1:r).
for (i = r:-1:3)
  % Apply rotation to L on the right.
  [c,s,L(r,i-1)] = gen_giv(L(r,i-1),L(r,i));
  L(r,i) = 0;                               % Eliminate L(r,i).
  [L(i-1:r-1,i-1),L(i-1:r-1,i)] = app_giv(L(i-1:r-1,i-1),L(i-1:r-1,i),c,s);
  [L(r+1:n,i-1),L(r+1:n,i)]     = app_giv(L(r+1:n,i-1),L(r+1:n,i),c,s);

  % Apply rotation to V on the right.
  if (vflag)
    [V(1:n,i-1),V(1:n,i)] = app_giv(V(1:n,i-1),V(1:n,i),c,s);
  end
end

% Flip last row of LA(1:r,1:r) down.
for (i = 1:r-1)
  % Restore LA to lower triangular form using rotation on the right.
  [c,s,LA(i,i)] = gen_giv(LA(i,i),LA(i,r));
  LA(i,r) = 0;                              % Eliminate LA(i,r).
  [LA(i+1:n,i),LA(i+1:n,r)] = app_giv(LA(i+1:n,i),LA(i+1:n,r),c,s);

  % Apply rotation to L on the left.
  [L(i,1:i+1),L(r,1:i+1)] = app_giv(L(i,1:i+1),L(r,1:i+1),c,s);

  % Apply rotation to UB on the right.
  if (uBflag)
    [UB(:,i),UB(:,r)] = app_giv(UB(:,i),UB(:,r),c,s);
  end
end

% Triangularize L using rotation on the right.
for (i = 1:r-1)
  [c,s,L(i,i)] = gen_giv(L(i,i),L(i,i+1));
  L(i,i+1) = 0;                             % Eliminate L(i,i+1).
  [L(i+1:n,i),L(i+1:n,i+1)] = app_giv(L(i+1:n,i),L(i+1:n,i+1),c,s);

  % Apply rotation to V on the right.
  if (vflag)
    [V(1:n,i),V(1:n,i+1)] = app_giv(V(1:n,i),V(1:n,i+1),c,s);
  end
end

%-----------------------------------------------------------------------
% End of function ullv_ref
%-----------------------------------------------------------------------
