function [LA,L,V,UA,UB] = ullv_rdef(LA,L,V,UA,UB,r,umin)
%  ullv_rdef --> Deflate one row of LA in the ULLV decomposition.
%
%  <Synopsis>
%    [LA,L,V,UA,UB] = ullv_rdef(LA,L,V,UA,UB,r,umin)
%
%  <Description>
%    Given the ULLV decomposition of the matrix pair A = UA*LA*L*V'
%    and B = UB*L*V', the function deflates LA(1:r,1:r). umin is an
%    estimate of the left singular vector of LA(1:r,1:r) associated
%    with the smallest singular value. On return, norm(LA(r,1:r)) is
%    of the order sigma_(r). The matrices UA, UB and V can be left out
%    by inserting an empty matrix [].
%
%  <See Also>
%    ullv_ref  --> Refine one row of LA in the ULLV decomposition.

%  <References>
%  [1] G.W. Stewart, "An Updating Algorithm for Subspace Tracking",
%      IEEE Trans. on SP 40 (1992), pp. 1535--1541.
%
%  [2] G.W. Stewart, "Updating a Rank-Revealing ULV Decomposition",
%      SIAM J. Matrix Anal. Appl., 14 (1993), pp. 494--499.
%
%  <Revision>
%    Ricardo D. Fierro, California State University San Marcos
%    Per Christian Hansen, IMM, Technical University of Denmark
%    Peter S.K. Hansen, IMM, Technical University of Denmark
%
%    Last revised: July 30, 2003
%-----------------------------------------------------------------------

% Check the input arguments.
vflag = 1;
if (isempty(V)), vflag = 0; end

uAflag = 1;
if (isempty(UA)), uAflag = 0; end

uBflag = 1;
if (isempty(UB)), uBflag = 0; end

% Initialize.
[n,n] = size(LA);

for (i = 2:r)
  % Transform umin to e_n.
  [c,s,umin(i)] = gen_giv(umin(i),umin(i-1));

  % Apply rotation to LA on the left.
  [LA(i-1,1:i),LA(i,1:i)] = app_giv(LA(i-1,1:i),LA(i,1:i),c,-s);

  % Apply rotation to UA on the right.
  if (uAflag)
    [UA(:,i-1),UA(:,i)] = app_giv(UA(:,i-1),UA(:,i),c,-s);
  end

  % Restore LA to lower triangular form using rotation on the right.
  [c,s,LA(i-1,i-1)] = gen_giv(LA(i-1,i-1),LA(i-1,i));
  LA(i-1,i) = 0;                            % Eliminate LA(i-1,i).
  [LA(i:n,i-1),LA(i:n,i)] = app_giv(LA(i:n,i-1),LA(i:n,i),c,s);

  % Apply rotation to L on the left.
  [L(i-1,1:i),L(i,1:i)] = app_giv(L(i-1,1:i),L(i,1:i),c,s);

  % Apply rotation to UB on the right.
  if (uBflag)
    [UB(:,i-1),UB(:,i)] = app_giv(UB(:,i-1),UB(:,i),c,s);
  end

  % Restore L to lower triangular form using rotation on the right.
  [c,s,L(i-1,i-1)] = gen_giv(L(i-1,i-1),L(i-1,i));
  L(i-1,i) = 0;                             % Eliminate L(i-1,i).
%%  [L(i:n,i-1),L(i:n,i)] = app_giv(L(i:n,i-1),L(i:n,i),c,s);   %% Old line.
  [L(i:end,i-1),L(i:end,i)] = app_giv(L(i:end,i-1),L(i:end,i),c,s);

  % Apply rotation to V on the right.
  if (vflag)
    [V(1:n,i-1),V(1:n,i)] = app_giv(V(1:n,i-1),V(1:n,i),c,s);
  end
end

%-----------------------------------------------------------------------
% End of function ullv_rdef
%-----------------------------------------------------------------------