function [u1,q1,flag_csne] = ullv_csne(A,LA,L,V,kappa)
%  ullv_csne --> Corrected semi-normal equations expansion (ULLV).
%
%  <Synopsis>
%    [u1,q1,flag_csne] = ullv_csne(A,LA,L,V,kappa)
%
%  <Description>
%    Compute the first row u1 of the m-by-n matrix UA and the first element
%    q1 of the expanded column q which is orthogonal to the columns of UA, by
%    using the LINPACK approach if it is safe, and if not, by solving the
%    following least squares problem by means of the CSNE method:
%
%              (A*V)'*(A*V)*z = (LA*L)'*(LA*L)*z = (A*V)'*e1
%
%    where
%
%              A = UA*LA*L*V'
%
%    If the parameter flag_csne is true, the CSNE approach has been used.
%    The parameter kappa (greater than one) is used to control the
%    orthogonalization procedure. A typical value for kappa is sqrt(2).
%
%  <Algorithm>
%    The algorithm is based on triangular solves. If LA is rank deficient,
%    then the rank information is used in the triangular solves.
%
%  <See Also>
%    mgsr     --> Modified Gram-Schmidt expansion.
%    ulv_csne --> Corrected semi-normal equations expansion (ULV).

%  <References>
%  [1] A. Bjorck, H. Park and L. Elden, "Accurate Downdating of Least Squares
%      Solutions", SIAM J. Matrix. Anal. Appl., 15 (1994), pp. 549--568.
%
%  [2] H. Park and L. Elden, "Downdating the Rank Revealing URV Decomposition",
%      SIAM J. Matrix Anal. Appl., 16 (1995), pp. 138--155.
%
%  <Revision>
%    Ricardo D. Fierro, California State University San Marcos
%    Per Christian Hansen, IMM, Technical University of Denmark
%    Peter S.K. Hansen, IMM, Technical University of Denmark
%
%    Last revised: June 22, 1999
%-----------------------------------------------------------------------

[m,n] = size(A);

% CSNE tolerances.
tau_csne  = 0.95;     % Tolerance in switch between Linpack and CSNE approach.
tol_csne  = 10*n*eps; % Tolerance in rank decision.
flag_csne = 0;

% Near-rank deficiency is not revealed in LA.
idx     = find(abs(diag(LA)) >  tol_csne);
idx_ill = find(abs(diag(LA)) <= tol_csne);

% z' = e1'*A*V.
z = (A(1,1:n)*V)';
z(idx_ill) = zeros(length(idx_ill),1);

% Solve L'*y = z. The right-hand side is overwritten by the solution.
z(idx) = L(idx,idx)'\z(idx);

% Solve LA'*UA(1,:)' = y. The right-hand side is overwritten by the solution.
z(idx) = LA(idx,idx)'\z(idx);

% Move solution to 1. row of UA, and compute q1.
u1 = z';

% Check whether Linpack approach is safe.
u1norm = norm(u1);
if (u1norm < tau_csne)
  q1 = sqrt(1 - u1norm^2);
  return;
end

% Use CSNE approach.
flag_csne = 1;

% Solve LA*y = UA(1,:)'. The right-hand side is overwritten by the solution.
z(idx) = LA(idx,idx)\z(idx);

% Solve L*z = y. The right-hand side is overwritten by the solution.
z = L\z;

% Find q = e1 - A*V*z.
q = eye(m,1) - A*(V*z);

% 2-norm of first solution q.
q1norm = norm(q);

% Solution refinement step...

z = ((q'*A)*V)';

% Solve L'*y = z. The right-hand side is overwritten by the solution.
z(idx) = L(idx,idx)'\z(idx);

% Solve LA'*dUA(1,:)' = y. The right-hand side is overwritten by the solution.
z(idx) = LA(idx,idx)'\z(idx);

% Correct 1. row of UA by solution.
u1 = u1 + z';

% Solve LA*y = dUA(1,:)'. The right-hand side is overwritten.
z(idx) = LA(idx,idx)\z(idx);

% Solve L*dz = y. The right-hand side is overwritten.
z = L\z;

q = q - A*(V*z);

% 2-norm of second solution q.
q2norm = norm(q);

if (q2norm <= q1norm/kappa)
  % We can orthogonalize any vector to UA(2:m,:).
  q1 = 0.0;
else
  % Normalization of the expanted column of UA.
  q1 = q(1)/q2norm;
end

%-----------------------------------------------------------------------
% End of function ullv_csne
%-----------------------------------------------------------------------
