function x_trrqr = trrqr(Q,R,Pi,p,b)
%  trrqr --> Solves a least squares problem using the RRQR decomposition.
%
%  <Synopsis>
%    x_trrqr = trrqr(Q,R,Pi,p,b)
%
%  <Description>
%    Solves the near-rank deficient least squares problem
%
%           min_x || b-A*x ||_2
%
%    using the RRQR decomposition. Here, A*Pi = Q*R is the RRQR
%    decomposition of A, p is the numerical rank of A, and the
%    TRRQR solution is defined by
%
%           x_trrqr = Pi(:,1:p)*inv(R(1:p,1:p))*Q(:,1:p)'*b.
%
%  <See Also>
%    tulv --> Solves a least squares problem using the ULV decomposition.
%    turv --> Solves a least squares problem using the URV decomposition.

%  <Revision>
%    Ricardo D. Fierro, California State University San Marcos
%    Per Christian Hansen, IMM, Technical University of Denmark
%    Peter S.K. Hansen, IMM, Technical University of Denmark
%
%    Last revised: June 22, 1999
%-----------------------------------------------------------------------

% Need the pseudoinverse of R(1:p,:); get it via QR factorization.
[P,RR] = qr(R(1:p,:)',0);

% Transform right hand side.
beta = Q(:,1:p)'*b;

% Back-substitution.
y = (RR')\beta;

% Compute TRRQR solution.
x_trrqr = Pi*(P*y);

%-----------------------------------------------------------------------
% End of function trrqr
%-----------------------------------------------------------------------
