function [smin,vmin] = inviter(R,max_iter,guess_v)
%  inviter --> Singular value/vector estimates via inverse iteration.
%
%  <Synopsis>
%    [smin,vmin] = inviter(R,max_iter,guess_v)
%
%  <Description>
%    Compute estimates smin and vmin of the smallest singular value
%    and corresponding right singular vector of the upper triangular
%    matrix R via inverse iteration using max_iter iterations. The
%    vector guess_v is the initial guess.
%
%  <Input parameters>
%    1. R        --> upper triangular matrix;
%    2. max_iter --> maximum number of steps of inverse iteration;
%    3. guess_v  --> initial guess vector;
%
%  <See Also>
%    ccvl --> Singular value/vector estimates via condition estimation.

%  <References>
%  [1] G.H. Golub and C.F. Van Loan, "Matrix Computations", Johns Hopkins
%      University Press, 3. Ed., p. 362, 1996.
%
%  <Revision>
%    Ricardo D. Fierro, California State University San Marcos
%    Per Christian Hansen, IMM, Technical University of Denmark
%    Peter S.K. Hansen, IMM, Technical University of Denmark
%
%    Last revised: June 22, 1999
%-----------------------------------------------------------------------

[n,n]  = size(R);
thresh = 1e-12;         % Relative stopping criterion for singular values.

% If R is a scalar.
if (n == 1)
  vmin = R/abs(R);
  smin = abs(R);
  return;
end

% Fix very small diagonal entries of R.
tol = sqrt(n)*norm(R,1)*eps;
p   = find(abs(diag(R)) < tol);
for (i = 1:length(p))
  R(p(i),p(i)) = tol;
end

% Initialize.
v = guess_v/norm(guess_v);

smin_old = norm(R*v);

% Perform inverse iterations.
for (i = 1:max_iter)
  u = R'\v;
  u = u/norm(u);
  v = R\u;
  v = v/norm(v);

  smin = norm(R*v);

  % Check for convergence of the singular value estimates.
  check = abs(smin - smin_old)/smin;
  if (check < thresh)
    break;
  else
    smin_old = smin;
  end
end

vmin = v;

%-----------------------------------------------------------------------
% End of function inviter
%-----------------------------------------------------------------------
