function hulvdemo
%  hulvdemo --> Demonstrates high-rank ULV algorithms.

%  <Revision>
%    Ricardo D. Fierro, California State University San Marcos
%    Per Christian Hansen, IMM, Technical University of Denmark
%    Peter S. K. Hansen, IMM, Technical University of Denmark
%
%    Last revised: January 2, 2005
%-----------------------------------------------------------------------

fprintf(1,'.                                                           \n');
fprintf(1,'.  The purpose of HULVDEMO is to demonstrate                \n');
fprintf(1,'.  the high-rank ULV algorithms in HULV and                 \n');
fprintf(1,'.  HULV_A.                                                  \n');
fprintf(1,'.                                                           \n');
input('.                                 [Press RETURN to continue]');
fprintf(1,'.                                                           \n');

format short e;

% Test matrix generation.
m = 50;
n = 20;
randn('seed',100);
A = randn(m,n);
numrank = 13;

[u,s,v] = svd(A);

s1 = 2*logspace(1,-3,numrank);
s2 = 5*logspace(-4,-6,n-numrank);
s(1:n,:) = diag([s1 s2]);

A = u*s*v';

% Define input parameters.
tol_rank = 0.001;
tol_ref  = 1e-04;
max_ref  = 1;

fprintf(1,'.                                                           \n');
fprintf(1,'.  Input parameters to HULV                                 \n');
fprintf(1,'............................................................\n');
fprintf(1,'.    no. rows m of A                     = %3.0f \n',m);
fprintf(1,'.    no. cols n of A                     = %3.0f \n',n);
fprintf(1,'.    rank tolerance (tol_rank)           = %6.4e \n',tol_rank);
fprintf(1,'.    refinement tolerance (tol_ref)      = %6.4e \n',tol_ref);
fprintf(1,'.    max refinement steps (max_ref)      = %3.0f \n',max_ref);
fprintf(1,'............................................................\n');
fprintf(1,'.                                                           \n');
fprintf(1,'.                                                           \n');
fprintf(1,'.  Compute the rank-revealing ULV decomposition using       \n');
fprintf(1,'.  Stewart''s algorithm:                                    \n');
fprintf(1,'.                                                           \n');
fprintf(1,'.    [p,L,V,U,vec] = hulv(A,tol_rank,tol_ref,max_ref)       \n');
fprintf(1,'.                                                           \n');
input('.                                 [Press RETURN to continue]');
fprintf(1,'.                                                           \n');
fprintf(1,'.  wait...                                                  \n');
fprintf(1,'.                                                           \n');

% Compute ULV.
t0 = cputime;

[p,L,V,U,vec] = hulv(A,tol_rank,tol_ref,max_ref);

time = cputime - t0;

lnorm = norm(L(p+1:n,1:p));
theta = norm(V(:,1:p)'*v(:,p+1:n));
phi   = norm(U(:,1:p)'*u(:,p+1:m));

fprintf(1,'.                                                           \n');
fprintf(1,'.  Output results from HULV                                 \n');
fprintf(1,'............................................................\n');
fprintf(1,'.    numerical rank p of A               = %3.0f \n',p);
fprintf(1,'.    upper bound of ||L(p+1:n,1:p)||     = %6.4e \n',vec(1));
fprintf(1,'.    estimate of pth singular value      = %6.4e \n',vec(2));
fprintf(1,'.    estimate of (p+1)th singular value  = %6.4e \n',vec(3));
fprintf(1,'.    estimate of num. nullspace angle    = %6.4e \n',vec(4));
fprintf(1,'.    estimate of num. range angle        = %6.4e \n',vec(5));
fprintf(1,'............................................................\n');
fprintf(1,'.                                                           \n');
fprintf(1,'.                                                           \n');
fprintf(1,'.  SVD based results                                        \n');
fprintf(1,'............................................................\n');
fprintf(1,'.    numerical rank p of A               = %3.0f \n',numrank);
fprintf(1,'.    ||L(p+1:n,1:p)||                    = %6.4e \n',lnorm);
fprintf(1,'.    pth singular value                  = %6.4e \n',s1(numrank));
fprintf(1,'.    (p+1)th singular value              = %6.4e \n',s2(1));
fprintf(1,'.    numerical nullspace angle           = %6.4e \n',theta);
fprintf(1,'.    numerical range angle               = %6.4e \n',phi);
fprintf(1,'............................................................\n');
fprintf(1,'.                                                           \n');
fprintf(1,'.                                                           \n');
fprintf(1,'.  Computational complexity of HULV                         \n');
fprintf(1,'............................................................\n');
fprintf(1,'.    CPU time                            = %6.4e \n',time);
fprintf(1,'............................................................\n');
fprintf(1,'.                                                           \n');
input('.                                 [Press RETURN to continue]');
fprintf(1,'.                                                           \n');

% Define input parameters.
tol_rank = 0.001;
max_iter = 5;
tol_ref  = 1e-04;
max_ref  = 1;

fprintf(1,'.                                                           \n');
fprintf(1,'.  Input parameters to HULV_A                               \n');
fprintf(1,'............................................................\n');
fprintf(1,'.    no. rows m of A                     = %3.0f \n',m);
fprintf(1,'.    no. cols n of A                     = %3.0f \n',n);
fprintf(1,'.    rank tolerance (tol_rank)           = %6.4e \n',tol_rank);
fprintf(1,'.    no. inverse iterations (max_iter)   = %3.0f \n',max_iter);
fprintf(1,'.    refinement tolerance (tol_ref)      = %6.4e \n',tol_ref);
fprintf(1,'.    max refinement steps (max_ref)      = %3.0f \n',max_ref);
fprintf(1,'............................................................\n');
fprintf(1,'.                                                           \n');
fprintf(1,'.                                                           \n');
fprintf(1,'.  Compute the rank-revealing ULV decomposition using the   \n');
fprintf(1,'.  alternative algorithm:                                   \n');
fprintf(1,'.                                                           \n');
fprintf(1,'.    [p,L,V,U,vec] = hulv_a(A,tol_rank,num_iter, ...        \n');
fprintf(1,'.                                         tol_ref,max_ref)  \n');
fprintf(1,'.                                                           \n');
input('.                                 [Press RETURN to continue]');
fprintf(1,'.                                                           \n');
fprintf(1,'.  wait...                                                  \n');
fprintf(1,'.                                                           \n');

% Compute ULV.
t0 = cputime;

[p,L,V,U,vec] = hulv_a(A,tol_rank,max_iter,tol_ref,max_ref);

time = cputime - t0;

lnorm = norm(L(p+1:n,1:p));
theta = norm(V(:,1:p)'*v(:,p+1:n));
phi   = norm(U(:,1:p)'*u(:,p+1:m));

fprintf(1,'.                                                           \n');
fprintf(1,'.  Output results from HULV_A                               \n');
fprintf(1,'............................................................\n');
fprintf(1,'.    numerical rank p of A               = %3.0f \n',p);
fprintf(1,'.    upper bound of ||L(p+1:n,1:p)||     = %6.4e \n',vec(1));
fprintf(1,'.    estimate of pth singular value      = %6.4e \n',vec(2));
fprintf(1,'.    estimate of (p+1)th singular value  = %6.4e \n',vec(3));
fprintf(1,'.    estimate of num. nullspace angle    = %6.4e \n',vec(4));
fprintf(1,'.    estimate of num. range angle        = %6.4e \n',vec(5));
fprintf(1,'............................................................\n');
fprintf(1,'.                                                           \n');
fprintf(1,'.                                                           \n');
fprintf(1,'.  SVD based results                                        \n');
fprintf(1,'............................................................\n');
fprintf(1,'.    numerical rank p of A               = %3.0f \n',numrank);
fprintf(1,'.    ||L(p+1:n,1:p)||                    = %6.4e \n',lnorm);
fprintf(1,'.    pth singular value                  = %6.4e \n',s1(numrank));
fprintf(1,'.    (p+1)th singular value              = %6.4e \n',s2(1));
fprintf(1,'.    numerical nullspace angle           = %6.4e \n',theta);
fprintf(1,'.    numerical range angle               = %6.4e \n',phi);
fprintf(1,'............................................................\n');
fprintf(1,'.                                                           \n');
fprintf(1,'.                                                           \n');
fprintf(1,'.  Computational complexity of HULV_A                      \n');
fprintf(1,'............................................................\n');
fprintf(1,'.    CPU time                            = %6.4e \n',time);
fprintf(1,'............................................................\n');
fprintf(1,'.                                                           \n');

%-----------------------------------------------------------------------
% End of function hulvdemo
%-----------------------------------------------------------------------
