function [beta,v,r] = gen_hous(x)
%  gen_hous --> Determine a Householder transformation.
%
%  <Synopsis>
%    [beta,v,r] = gen_hous(x)
%
%  <Description>
%    Given a vector x, this function computes the scalar beta and
%    the vector v such that (I - beta*v*v')x is zero in all but the
%    first component r = -sign(x1)*norm(x). If x = 0 then v = 0 and
%    beta = 1 is returned.
%
%  <See Also>
%    app_hous --> Apply a Householder transformation.

%  <References>
%  [1] G.H. Golub and C.F. Van Loan, "Matrix Computations", Johns Hopkins
%      University Press, 3. Ed., p. 210, (1996).
%
%  <Revision>
%    Ricardo D. Fierro, California State University San Marcos
%    Per Christian Hansen, IMM, Technical University of Denmark
%    Peter S.K. Hansen, IMM, Technical University of Denmark
%
%    Last revised: June 22, 1999
%-----------------------------------------------------------------------

r = -(sign(x(1)) + (x(1)==0)) * norm(x);    % Modification for sign(0)=1.
v = x;                                      % v(2:n) = x(2:n).

if (r == 0)                                 % If x is the zero vector.
  beta = 1;
else
  v(1) = x(1) - r;                          % v1 = x1 + sign(x1)*norm(x).
  beta = 1/(-r'*v(1));                      % beta = 2/v'*v.
end

%-----------------------------------------------------------------------
% End of function gen_hous
%-----------------------------------------------------------------------
