(* :Title: datashfv6.m   *)

(* :Context: shohatfavard`datashfv6` *)

(* :Author:  Zlia da ROCHA *)

(* :Address: 
            Departamento de Matemtica Aplicada da
            Faculdade de Cincias do Porto
            Rua das Taipas, 135
            4050 Porto
            Portugal
  
            Tel: ++351 02 2080313
            Fax: ++351 02 2004109
            Email: mrdioh@fc.up.pt
*)
(* :Summary:
    The datashfv6 package contains the definitions 
    gamma, beta and formexact that the package shfv 
    needs, corresponding to the example 6 of the first
    paper cited bellow.    
*)
(* :Date: July 1999 *)

(* :Available at http://netlib.bell-labs.com/ in the 
    numeralgo library in netlib:       *)

(* :Package Version: 1.0 *)

(* :Mathematica Version: 3.0 *)  
 
(* :Keywords: Shohat Favard's method, regular forms,
              d-dimensional canonical forms, 
              d-orthogonal polynomial sequences.
*)
(* :Sources:

 - Z. da Rocha, Shohat-Favard and Chebyshev's
   methods in d-orthogonality, Numerical Algorithms,
   20 (1999) 139 - 164.
   
 - About the example treated here see pp.286,293,294 
 of:
   K. Douak, The relation of the d-orthogonal
   polynomials to the Appell polynomials,  
   J. Comput. Appl. Math., 70 (1996) 279-295.
*)

(* :Warnings: How to load this package.

This package must be putted in a directory named 
shohatfavard. 
Note that the directory containing the shohatfavard
directory must be on the file search path $Path.

This package contains the definitions needed in the
package shfv.m. 
The context corresponding to this package 
(shohatfavard`datashfv6`) must be written in the
BeginPackage command of the package shfv.m
Then, when you load the package shfv.m doing 
<<shohatfavard`shfv`or Needs["shohatfavard`shfv`"]
this package is automatically loaded. 
*)

BeginPackage["shohatfavard`datashfv6`"]

datashfv6::usage="is a package that contains
the definitions of

- the d gamma coefficients and the beta coefficient
  that appear in the (d+1)-order recurrence relation
  satisfied by orthogonal polynomials of dimension d
  (gamma, beta)
  
- the exact expressions of the moments of the d
  dimensional form, with respect to which the
  polynomials are orthogonal of dimension d
  (formexact)

, corresponding to the example 6 of the paper
  
Z. da ROCHA, Shohat-Favard and Chebyshev's
             methods in d-orthogonality,
             Numerical Algorithms 20 (1999) 139 - 164.
             
The definitions gamma, beta and formexact are used in
the package shfv.m."


gamma::usage="
gamma[ d, r, n ], is the n-th gamma coefficient
of order r (n>=1, 0<=r<=d-1), that appears in the
(d+1)-order recurrence relation satisfied by
orthogonal polynomials of dimension d.

gamma is used in the package shfv.m."


beta::usage="
beta[ d, n ], n>=0, is the n-th beta coefficient, 
that appears in the (d+1)-order recurrence relation
satisfied by orthogonal polynomials of dimension d.

beta is used in the package shfv.m."


formexact::usage="
formexact[ d, alpha, n ] is the exact expression
of the n-th moment (n>=0) of the form of order
alpha (1<=alpha<=d), with respect to which the
polynomials are orthogonal of dimension d.

formexact is used in the package shfv.m." 


Begin["`Private`"]

(*___________________________________________________

                   Error Messages
 ___________________________________________________*)
 
gamma::badarg2=
   "wrong value to the second argument r=`1` in gamma; 
    r must be an integer and verify 0<=r<=`2`;
    try again"

gamma::badarg3=
   "wrong value to the third argument n=`1` in gamma;
    n must be an integer;
    try again" 

beta::badarg2=
   "wrong value to the second argument n=`1` in beta;
    n must be an integer;
    try again"
    
formexact::badarg2=
   "wrong value to the second argument
    alpha=`1` in formexact;
    alpha must be an integer and verify
    1<=alpha<=`2`;
    try again"

formexact::badarg3=
   "wrong value to the third argument n=`1` in formexact;
    n must be an integer;
    try again"  

(*______________________________________________

               Exported definitions
________________________________________________*)
(*______________________________________________

              the gamma coefficients
_______________________________________________*)

gamma[d_,r_,n_]:=gamma[d,r,n]=

Block[{},
        
(*___________ checking the arguments____________*)
  
  If[Or[r<0,r>=d,Not[IntegerQ[r]]],
  
  Return[Message[gamma::badarg2,r,d-1]]
    ];
    
  If[Not[IntegerQ[n]],
  
    Return[Message[gamma::badarg3,n]]
    ];
    
(*______________________________________________*)
		
 Which[n<=0,Return[0],
       r>0,Return[0],
       And[r===0,n===1],
         Return[ 1/(d+1) ],
       And[r===0,n>=2],
         Return[ (n-1+d)/(n-1)*gamma[d,0,n-1] ]
      ](*end of Which[n]*)
(*_____________________________________________*)
   
]; (*end of Block*)
(*_____________________________________________*) 
   
(*______________________________________________

              the beta coefficients
_______________________________________________*)

beta[d_,n_]:=beta[d,n]=

 Block[{ro1},
 
 (*___________ checking the arguments___________*)
      
      If[Not[IntegerQ[n]],
      
     Return[Message[beta::badarg3,n]]
        ];  
 (*_____________________________________________*) 
 
 Return[0]
 
(*_____________________________________________*)
   
  ]; (*end of Block*)          
         
(*________________________________________________

    the exact expression of the moments of the 
    d-dimensional form
_________________________________________________*)

formexact[d_,alpha_,n_]:=formexact[d,alpha,n]=

 Block[{ro1,m3,fact},
 (*___________ checking the arguments___________*)
   If[Or[alpha<1,alpha>d,Not[IntegerQ[alpha]]],
    
 	  Return[Message[formexact::badarg2,alpha,d]]
 	 ];	 
   
   If[Not[IntegerQ[n]],
   
      Return[Message[formexact::badarg3,n]]
     ];
(*_______________________________________________*)
                 
Which[n<0, Return[0],
      alpha===1, (* moments of the first form *)
          If[n===0,Return[1]];
          m=Floor[n/(d+1)];r=n-m*(d+1);
          Which[ r>0,Return[0],
                 And[r===0,m===1], Return[1/(d+1)],
                 And[r===0,m>1],
 Return[Binomial[n-1,d]*formexact[d,1,n-(d+1)]/(d+1)]
               ],(*end of Which*)
        alpha>1,
          Which[ And[n>=0,n<alpha-1], Return[0],
                 n===alpha-1,Return[1],
                 n>=alpha,
     Return[n*formexact[d,alpha-1,n-1]/(alpha-1)]
               ](*end of Which*)
       ](*end of Which[alpha]*)    
(*______________________________________________*)
        
];(*end of Block*) 

(*______________________________________________*)
   
   
End[]

EndPackage[]