(* :Title: datamodchv43.m   *)

(* :Context: modchebyshev`datamodchv43` *)

(* :Author:  Zlia da ROCHA *)

(* :Address: 
            Departamento de Matemtica Aplicada da
            Faculdade de Cincias do Porto
            Rua das Taipas, 135
            4050 Porto
            Portugal
  
            Tel: ++351 02 2080313
            Fax: ++351 02 2004109
            Email: mrdioh@fc.up.pt
*)
(* :Summary:
    The datamodchv43 package contains the definition
    form, auxgamma, auxbeta, gammaexact and betaexact
    that the package modchv needs, corresponding to 
    the example 4.3 of the first paper cited bellow.    
*)
(* :Date: July 1999. *)

(* :Available at http://netlib.bell-labs.com/ in the 
    numeralgo library in netlib:       *)

(* :Package Version: 1.0 *)

(* :Mathematica Version: 3.0 *)  
 
(* :Keywords: Modified Chebyshev's method, recurrence
              relations, orthogonal polynomials of 
              dimension d, d-dimensional forms.
*)
(* :Sources:

 - Z. da Rocha, Shohat-Favard and Chebyshev's
   methods in d-orthogonality, Numerical Algorithms,
   20 (1999) 139-164.
   
 - About the example treated here see pp.82, 103 of
 
   K. Douak, P. Maroni, Les polynmes orthogonaux 
   "classiques" de dimension deux, Analysis 12, 
   71-107 (1992),

   and p.34 of
   
   K. Douak, P. Maroni, On d-orthogonal Tchebychev 
   polynomials, I, Appl. Numer. Math., 24 (1997) 23-53.
   
*)

(* :Warnings: How to load this package.

This package must be putted in a directory named 
modchebyshev. 
Note that the directory containing the modchebyshev
directory must be on the file search path $Path.

This package contains the definitions needed in the
package modchv.m. 
The context corresponding to this package 
(modchebyshev`datamodchv43`) must be written in the
BeginPackage command of the package modchv.m.
Then, when you load the package modchv.m doing 
<<modchebyshev`modchv`or Needs["modchebyshev`modchv`"],
in any notebook, this package is automatically loaded. 
*)


BeginPackage["modchebyshev`datamodchv43`"]

datamodchv43::usage="is a package that contains
the definitions of:

- the moments of the d forms with respect to which
  the polynomials are orthogonal of dimension d
  (form),
  
- the d gamma coefficients and the beta coefficient
  of the recurrence relation of the auxiliary family
  of orthogonal polynomials of dimension d 
  (auxgamma, auxbeta),
  
- the exact expression of the d gamma coefficients
  and the beta coefficient that appear in the 
  (d+1)-order recurrence relation satisfied by 
  orthogonal polynomials of dimension d, in the case
  they are known (gammaexact, betaexact);
  
corresponding to the example 4.3 of the paper

Z. da Rocha, Shohat-Favard and Chebyshev's
methods in d-orthogonality, Numerical Algorithms,
20 (1999) 139-164.

The definitions form, auxgamma, auxbeta, gammaexact 
and betaexact are used in the package modchv.m."

form::usage="
form[d, alpha, n] is the moment of order n (n>=0)
of the form of order alpha (1<=alpha<=d), with
respect to which the polynomials are orthogonal
of dimension d.
form is used in the package modchv.m."

auxgamma::usage="
auxgamma[ d, r, n ], n>=1,
is the n-th gamma coefficient of order r (0<=r<=d-1),
that appears in the (d+1)-order recurrence relation
satisfied by the auxiliary sequence of orthogonal
polynomials of dimension d.
auxgamma is used in the package modchv.m."


auxbeta::usage="
auxbeta[ d, n ], n>=1,
is the n-th beta coefficient that appears in the 
(d+1)-order recurrence relation satisfied by the
auxiliary sequence of orthogonal polynomials of
dimension d.
auxbeta is used in the package modchv.m."


gammaexact::usage="
gammaexact[ d, r, n ], is the exact expression of
the n-th gamma coefficient of order r (n>=1,
0<=r<=d-1), that appears in the (d+1)-order recurrence
relation satisfied by orthogonal polynomials of
dimension d.
gammaexact is used in the package modchv.m."

betaexact::usage="
betaexact[ d, n ], n>=0, is the exact expression of
the n-th beta coefficient, that appears in the 
(d+1)-order recurrence relation satisfied by
orthogonal polynomials of dimension d.
betaexact is used in the package modchv.m."

Begin["`Private`"]

(*___________________________________________

                Error Messages
 __________________________________________*)
   
form::badarg2=
   "wrong value to the second argument
    alpha=`1` in form;
    alpha must be an integer and verify
    1<=alpha<=`2`;
    try again"

form::badarg3=
   "wrong value to the third argument n=`1` in form;
    n must be an integer;
    try again" 

auxgamma::badarg2=
   "wrong value to the second argument r=`1`
    in auxgamma; r must be an integer and
    verify 0<=r<=`2`;
    try again"

auxgamma::badarg3=
   "wrong value to the third argument n=`1`
    in auxgamma;
    n must be an integer;
    try again" 

auxbeta::badarg2=
   "wrong value to the second argument n=`1`
    in auxbeta;
    n must be an integer;
    try again" 

gammaexact::badarg2=
   "wrong value to the second argument r=`1`
    in gammaexact; r must be an integer and
    verify 0<=r<=`2`;
    try again"

gammaexact::badarg3=
   "wrong value to the third argument n=`1`
    in gammaexact;
    n must be an integer;
    try again" 

betaexact::badarg2=
   "wrong value to the second argument n=`1`
    in betaexact;
    n must be an integer;
    try again" 

(*___________________________________________

             Exported definition
___________________________________________*)
(*___________________________________________

   the moments of the d-dimensional form
___________________________________________*)

form[d_,alpha_,n_]:=form[d,alpha,n]=

 Block[{fi,ld,m3,m,s,gamma01},
 (*_________ checking the arguments _________*)
 
   If[Or[alpha<1,alpha>d,Not[IntegerQ[alpha]]],
    
 	  Return[Message[form::badarg2,alpha,d]]
 	 ];	 
   
   If[Not[IntegerQ[n]],
   
      Return[Message[form::badarg3,n]]
     ];
(*_________________________________________*)

    Which[n<0,Return[0], 
          n>=0,   
      gamma01=1; fi=1; ld=1;
      m=Floor[n/3]; m3=3*m; s=n-m3;
          Which[
                alpha===1,
                Which[n===0,Return[1],
                      s===0,
Return[ gamma01*(fi+2)*(ld+3)*(m3-1)*(m3-2)/
        2/(fi+m3-1)/(ld+m3)*form[2,1,m3-3] ],
                      Or[s===1,s===2],
               Return[0] ](*end of Which[s]*),
                alpha===2,
                Which[Or[s===0,s===2],Return[0],
                      n===1,Return[1],
                      s===1,
Return[ gamma01*(fi+2)*(ld+3)*(m3+1)*(m3-1)/
         2/(fi+m3+2)/(ld+m3)*form[2,2,m3-2] ]
                      ](*end of Which[s]*)
                ](*end of Which[alpha]*)
        ](*end of Which[n]*)  
(*_____________________________________________*)
   
]; (*end of Block*)  
 
(*_______________________________________________

the gamma coefficients of the auxiliary sequence 
of d-orthogonal polynomials
________________________________________________*)

auxgamma[d_,r_,n_]:=auxgamma[d,r,n]=

 Block[{const},
 (*_________ checking the arguments __________*)  
  If[Or[r<0,r>=d,Not[IntegerQ[r]]],
  
  Return[Message[auxgamma::badarg2,r,d-1]]
    ];
    
  If[Not[IntegerQ[n]],
  
    Return[Message[auxgamma::badarg3,n]]
    ];
 (*____________________________________________*)
 
  Which[n<=0,Return[0],
        n>0,
        const=1;
          Which[r===0, 
                 Return[const*n*(n+5)/(n+2)/(n+3)],
                r===1, 
                 Return[0]
               ] (*end of Which[r]*)
       ](*end of Which[n]*)
        
(*_____________________________________________*)        
        
 ];(*end of Block*) 
 
(*___________________________________________
   
the beta coefficients of the auxiliary sequence 
of d-orthogonal polynomials
_____________________________________________*)

auxbeta[d_,n_]:=auxbeta[d,n]=

 Block[{},
 (*_________ checking the arguments __________*)
  If[Not[IntegerQ[n]],
  
    Return[Message[auxbeta::badarg2,n]]
    ];
(*____________________________________________*) 

 Return[0]
 
(*____________________________________________*)
        
 ];(*end of Block*)  
 
(*_________________________________________________

  the exact expression of the gamma coefficients
_________________________________________________*) 

gammaexact[d_,r_,n_]:=gammaexact[d,r,n]=

 Block[{gamma01,fi,ld,s,m,m3},
 (*_________ checking the arguments __________*) 
 
  If[Or[r<0,r>=d,Not[IntegerQ[r]]],
  
  Return[Message[gammaexact::badarg2,r,d-1]]
    ];
    
  If[Not[IntegerQ[n]],
  
    Return[Message[gammaexact::badarg3,n]]
    ];
(*____________________________________________*)

     Which[n<=0,Return[0],
           n>0,
    gamma01=1; fi=1; ld=1;
    m=Floor[n/2]; s=n-2*m;
    m3=3*m;
    Which[ r===0,
           Which[s===0,
Return[ gamma01*(fi+2)*(ld+3)*m*(2*m+1)*(m+fi)/
        (m3+fi-1)/(m3+fi+2)/(m3+ld) ],
                 s===1,
Return[ gamma01*(fi+2)*(ld+3)*(m+1)*(2*m+1)*(m+ld)/
        (m3+ld)/(m3+ld+3)/(m3+fi+2) ]         
                 ](*end of Which[s]*),
           r===1,Return[0]
          ](*end of Which[r]*)  
      ](*end of Which[n]*)    
        
(*_________________________________________________*)

];(*end of Block*)


(*_________________________________________________

  the exact expression of the beta coefficients
_________________________________________________*) 

betaexact[d_,n_]:=betaexact[d,n]=

 Block[{},
  (*_________ checking the arguments _________*)
      
      If[Not[IntegerQ[n]],
      
     Return[Message[betaexact::badarg2,n]]
        ];   
  (*__________________________________________*)
             
     Return[0] 
 (*_____________________________________________*) 
   
      ];(*end of Block*)          
(*_______________________________________________*)

   
End[]

EndPackage[]