(* :Title: datachv53.m   *)

(* :Context: chebyshev`datachv53` *)

(* :Author:  Zlia da ROCHA *)

(* :Address: 
            Departamento de Matemtica Aplicada da
            Faculdade de Cincias do Porto
            Rua das Taipas, 135
            4050 Porto
            Portugal
  
            Tel: ++351 02 2080313
            Fax: ++351 02 2004109
            Email: mrdioh@fc.up.pt
*)
(* :Summary:
    The datachv53 package contains the definition form,
    gammaexact and betaexact that the package chv needs,
    corresponding to the example 5.3 of the first paper
    cited bellow.    
*)
(* :Date: July 1999. *)

(* :Available at http://netlib.bell-labs.com/ in the 
    numeralgo library in netlib:       *)

(* :Package Version: 1.0 *)

(* :Mathematica Version: 3.0 *)  
 
(* :Keywords: chebyshev's method, recurrence relations, 
              orthogonal polynomials of dimension d,
              d-dimensional forms.
*)
(* :Sources:

 - Z. da Rocha, Shohat-Favard and Chebyshev's
   methods in d-orthogonality, Numerical Algorithms,
   20 (1999) 139-164.
   
 - About the example treated here see  p.286 of:
 
   K. Douak, The relation of the d-orthogonal
   polynomials to the Appell polynomials,  
   J. Comput. Appl. Math., 70 (1996) 279-295.
*)

(* :Warnings: How to load this package.

This package must be putted in a directory named
chebyshev. 

This package contains the definitions needed in the
package chv.m. 
The context corresponding to this package
(chebyshev`datachv53`) must be written in the 
BeginPackage command of the package chv.m.
Then, when you load the package chv.m doing 
<<chebyshev`chv` or Needs["chebyshev`chv`"] in any
notebook, this package is automatically loaded. 
*)


BeginPackage["chebyshev`datachv53`"]

datachv53::usage="is a package that contains
the definitions of:

- the moments of the d forms with respect to which
  the polynomials are orthogonal of dimension d
  (form),
  
- the exact expression of the d gamma coefficients
  and the beta coefficient that appear in the 
  (d+1)-order recurrence relation satisfied by 
  orthogonal polynomials of dimension d, in the case
  they are known (gammaexact, betaexact);
  
corresponding to the example 5.3 of the paper

Z. da Rocha, Shohat-Favard and Chebyshev's
methods in d-orthogonality, Numerical Algorithms,
20 (1999) 139-164.

The definitions form, gammaexact and betaexact are
used in the package chv.m."

form::usage="
form[d, alpha, n] is the moment of order n (n>=0)
of the form of order alpha (1<=alpha<=d), with
respect to which the polynomials are orthogonal
of dimension d.
form is used in the package chv.m."

gammaexact::usage="
gammaexact[ d, r, n ], is the exact expression of
the n-th gamma coefficient of order r (n>=1,
0<=r<=d-1), that appears in the (d+1)-order recurrence
relation satisfied by orthogonal polynomials of
dimension d.
gammaexact is used in the package chv.m."

betaexact::usage="
betaexact[ d, n ], n>=0, is the exact expression of
the n-th beta coefficient, that appears in the 
(d+1)-order recurrence relation satisfied by
orthogonal polynomials of dimension d.
betaexact is used in the package chv.m."

Begin["`Private`"]

(*___________________________________________

                Error Messages
 __________________________________________*)
   
gammaexact::badarg2=
   "wrong value to the second argument r=`1`
    in gammaexact; r must be an integer and
    verify 0<=r<=`2`;
    try again"

gammaexact::badarg3=
   "wrong value to the third argument n=`1`
    in gammaexact;
    n must be an integer;
    try again" 

betaexact::badarg2=
   "wrong value to the second argument n=`1`
    in betaexact;
    n must be an integer;
    try again"
    
form::badarg2=
   "wrong value to the second argument
    alpha=`1` in form;
    alpha must be an integer and verify
    1<=alpha<=`2`;
    try again"

form::badarg3=
   "wrong value to the third argument n=`1` in form;
    n must be an integer;
    try again"  

(*________________________________________________

                 Exported definition
_________________________________________________*)
(*_________________________________________________

  the exact expression of the gamma coefficients
_________________________________________________*) 

gammaexact[d_,r_,n_]:=gammaexact[d,r,n]=
 Block[{gamma1,g11,g12},
 
 (*_________ checking the arguments __________*) 
 
  If[Or[r<0,r>=d,Not[IntegerQ[r]]],
  
  Return[Message[gammaexact::badarg2,r,d-1]]
    ];
    
  If[Not[IntegerQ[n]],
  
    Return[Message[gammaexact::badarg3,n]]
    ];
(*____________________________________________*)

     g11=3/2;g12=4/3;
     gamma1={1/(d+1),g11,g12};
     Which[n<=0,Return[0],
           n===1,
            Return[gamma1[[r+1]] ],
           n>1,
            Return[(n-1+d-r)/(n-1)*
                   gammaexact[d,r,n-1]]
     ](*end of Which[n]*)    
        
(*_________________________________________________*)

];(*end of Block*)


(*_________________________________________________

  the exact expression of the beta coefficients
_________________________________________________*) 

betaexact[d_,n_]:=betaexact[d,n]=

 Block[{},
 
  (*_________ checking the arguments _________*)
      
      If[Not[IntegerQ[n]],
      
     Return[Message[betaexact::badarg3,n]]
        ];   
  (*__________________________________________*)
             
     Return[0] 
 (*_____________________________________________*) 
   
      ];(*end of Block*)          
(*______________________________________________*)

(*______________________________________________

   the moments of the d-dimensional form 
   calculated using the package shfv.m
________________________________________________*)

form[d_,alpha_,n_]:=
 Block[{},
 
 (*_________ checking the arguments _________*)
 
   If[Or[alpha<1,alpha>d,Not[IntegerQ[alpha]]],
    
 	  Return[Message[form::badarg2,alpha,d]]
 	 ];	 
   
   If[Not[IntegerQ[n]],
   
      Return[Message[form::badarg3,n]]
     ];
(*_________________________________________*)

Return[
{{1,0,4/3 ,3/2 ,67/12 ,20, 1135/18 ,2345/8 ,
    547505/432, 6055, 582505/18, 16807175/96,
    1773706165/1728, 19013995/3, 105207697595/2592,
    316053012275/1152, 39953736047225/20736,
    3033988582625/216, 1656935360155075/15552 ,
    3848941950728875/4608,                     
 5052791584924508425/746496, 
 293629599641753375/5184,
 182167180235827862075/373248,  
 2154094815020045572675 /497664,            
 353475770678943680993225/8957952,
   22968833625377031266875/62208,
   5952112645915296812826875/1679616,
     23124649259229756871758125 /663552,    
    4190253167288399409757934375/11943936,
    449484225421418341456176875/124416,
    681270677094040710152464113125 / 17915904, 
 
    3255097972826703136070528215625 / 7962624, 
 
    1928845897991794779773603363160625 / 429981696, 
 
    37483945518357427019442456319375 / 746496, 
 
    61596331624710481680263490978044375 / 107495424, 
 
    1909909628781517885103660133921490625 / 286654464, 
 
    407085632591346602450504657162449871875 / 5159780352, 
 
    11357975562161924462310263373018990625 / 11943936, 
 
    90255192169915538464040021270474943434375 / 
 
     7739670528, 5002527719999861179129395677935804558656\
 
      25 / 3439853568, 11415206364822781714304926955348985\
 
      6890996875 / 61917364224, 
 
    30634671474095567328312481361000769026140625 / 
 
     1289945088, 72150562057170297254246907575101909738476\
 
      15625 / 23219011584, 
 
    170410963550382929113423026399279055778004434375 / 
 
     41278242816, 1240932017337642536260311516233783919992\
 
      11527809375 / 2229025112064, 
 
    245659606994281242538241157225501300902553359375 / 
 
     322486272, 392800113542500899107631571145642694014073\
 
      5206390625 / 371504185344, 
 
    737157701339503533898486548721887399952860459156718\
 
      75 / 495338913792, 
 
    189331078079816459526050321451284718813030597203605156\
 
      25 / 8916100448256, 
 
    9506855261324818022037491921330456505508413574537656\
 
      25 / 30958682112, 
 
    300885235069681101273565762524302722118477989822490923\
 
      4375 / 6687075336192, 
 
    396963190831495641394722549435700602933698370326913378\
 
      90625 / 5944066965504, 
 
    107401283213870354325912955398538934293142902215246337\
 
      35671875 / 106993205379072, 
 
    340515610696937562998667106269947417506224952736819757\
 
      3171875 / 2229025112064, 
 
    377677772478306074962055749654387313252083992181398239\
 
      4665921875 / 160489808068608, 
 
    290760953266417939136376038676127170258248983681496732\
 
      4383515625 / 7925422620672, 
 
    222898886686804499153927702611676187970148277988313745\
 
      51957514765625 / 3851755393646592, 
 
    246994980812842487261727381920731989746948747947169429\
 
      7875998984375 / 26748301344768, 
 
    224220304909565563818543320459864564114295749011002838\
 
      18747463515625 / 15045919506432, 
 
    624491865238049694277484255055115865623246952178518937\
 
      74472586775703125 / 2567836929097728, 
 
    185464735922233724890317750729105388617543815561948971\
 
      56684389973379609375 / 46221064723759104},
      {0, 1, 0, 4, 6, 335/12, 120, 7945/18, 2345,
   547505/48, 60550, 
    6407555/18,  16807175/8,  23058180145/1728,
      266195930/3,                                 
  526038487975/864, 316053012275/72,  
  679213512802825/20736,
    3033988582625/12,  31481771842946425/15552,
      19244709753644375/1152,
    35369541094471558975/248832,
      3229925596059287125/2592, 
    4189845145424040827725/373248,
      2154094815020045572675/20736,   
    8836894266973592024830625/ 8957952, 
      298594837129901406469375/31104,
    5952112645915296812826875/62208,
    161872544814608298102306875/165888
    121517341851363582882980096875 / 11943936, 
 
    2247421127107091707280884375/20736,
   
    21119390989915262014726387506875 / 17915904, 
 
    3255097972826703136070528215625 / 248832, 
 
    21217304877909742577509636994766875 / 143327232, 
 
    637227073812076259330521757429375 / 373248, 
 
    2155871606864866858809222184231553125 / 107495424, 
 
    1909909628781517885103660133921490625 / 7962624, 
 
    15062168405879824290668672315010645259375 / 
 
     5159780352, 215801535681076564783895004087360821875 / 
 
     5971968, 11733174982089020000325202765161742646468\
 
      75 / 2579890176, 25012638599999305895646978389679022\
 
      79328125 / 429981696, 
 
    4680234609577340502865020051693084132530871875 / 
 
     61917364224, 2144427003186689712981873695270053831829\
 
      84375 / 214990848, 
 
    310247416845832278193261702572938211875447471875 / 
 
     23219011584, 1874520599054212220247653290392069613558\
 
      048778125 / 10319560704, 
 
    620466008668821268130155758116891959996057639046875 / 
 
     247669456896, 565017096086846857837954661618652992075\
 
      8727265625 / 161243136, 
 
    1846160533649754225805868384384520661866145547003593\
 
      75 / 371504185344, 
 
    737157701339503533898486548721887399952860459156718\
 
      75 / 10319560704, 
 
    927722282591100651677646575111295122183849926297665265\
 
      625 / 8916100448256, 
 
    237671381533120450550937298033261412637710339363441406\
 
      25 / 15479341056, 
 
    511504899618457872165061796291314627601412582698234569\
 
      84375 / 2229025112064, 
 
    516052148080944333813139314266410783813807881424987392\
 
      578125 / 1486016741376, 
 
    569226801033512877927338663612256351753657381740805587\
 
      990609375 / 106993205379072, 
 
    340515610696937562998667106269947417506224952736819757\
 
      3171875 / 41278242816, 
 
    207722774863068341229130662309913022288646195699769031\
 
      706625703125 / 160489808068608, 
 
    203532667286492557395463227073289019180774288577047712\
 
      70684609375 / 990677827584, 
 
    423507884704928548392462634962184757143281728177796116\
 
      487192780546875 / 1283918464548864, 
 
    716285444357243213059009407570122770266151369046791346\
 
      38403970546875 / 13374150672384, 
 
    132289979896643682652940559071320092827434491916491674\
 
      5306100347421875 / 15045919506432, 
 
    312245932619024847138742127527557932811623476089259468\
 
      872362933878515625 / 213986410758144},
      {0, 0, 1, 0, 8, 15, 335/4, 420,15890/9 ,21105/2,  
    2737525/48, 333025, 6407555/3, 218493275/16,
    161407261015/1728, 665489825, 
    526038487975/108,  5372901208675/144,
      679213512802825/2304,
    57645783069875/24,  157408859214732125/7776,
    
    134712968275510625/768,
      389064952039187148725/248832,
    
    74288288709363603875/5184,
      4189845145424040827725/31104,
    
    53852370375501139316875/41472,
      114879625470656696322798125/8957952,
    
    298594837129901406469375/2304,
      41664788521407077689788125/31104,
    
    4694303799623640644966899375/ 331776,
    
    607586709256817914414900484375 / 3981312, 
 
    69670054940319842925707415625 / 41472, 
 
    21119390989915262014726387506875 / 1119744, 
 
    35806077701093734496775810371875 / 165888, 
 
    360694182924465623817663828911036875 / 143327232, 
 
    22302947583422669076568261510028125 / 746496, 
 
    2155871606864866858809222184231553125 / 5971968, 
 
    70666656264916161748835424955095153125 / 15925248, 
 
    286181199711716661522704773985202259928125 / 
 
     5159780352, 28054199638539953421906350531356906843\
 
      75 / 3981312, 58665874910445100001626013825808713232\
 
      34375 / 644972544, 
 
    102551818259997154172152611397683993452453125 / 
 
     859963392, 327616422670413835200551403618515889277161\
 
      03125 / 20639121408, 
 
    9221036113702765765822056889661231476868328125 / 
 
     429981696, 341272158530415506012587872830232033062992\
 
      2190625 / 11609505792, 
 
    9372602995271061101238266451960348067790243890625 / 
 
     2293235712, 14270718199382889166993582436688515079909\
 
      325698078125 / 247669456896, 
 
    265558035160818023183838690960766906275660181484375 / 
 
     322486272, 184616053364975422580586838438452066186614\
 
      554700359375 / 15479341056, 
 
    36120727365635673161025840887372482597690162498679218\
 
      75 / 20639121408, 
 
    231930570647775162919411643777823780545962481574416316\
 
      40625 / 8916100448256, 
 
    404041348606304765936593406656544401484107576917850390\
 
      625 / 10319560704, 
 
    664956369503995233814580335178709015881836357507704940\
 
      796875 / 1114512556032, 
 
    273507638482900496920963836561197715421318177155243318\
 
      06640625 / 2972033482752, 
 
    569226801033512877927338663612256351753657381740805587\
 
      990609375 / 3962711310336, 
 
    187283585883315659649266908448471079628423724005250866\
 
      524453125 / 82556485632, 
 
    145405942404147838860391463616939115602052336989838322\
 
      1946379921875 / 40122452017152, 
 
    386712067844335859051380131439249136443471148296390654\
 
      143007578125 / 660451885056, 
 
    122817286564429279033814164139033579571551701171560873\
 
      78128590635859375 / 1283918464548864, 
 
    422608412170773495704815550466372434457029307737606894\
 
      3665834262265625 / 26748301344768, 
 
    661449899483218413264702795356600464137172459582458372\
 
      6530501737109375 / 2507653251072}}[[alpha]][[n+1]]
  ] (*end of Return*)
(*_____________________________________________*)
   
]; (*end of Block*)  

(*_______________________________________________*) 

   
End[]

EndPackage[]